// Copyright 2024 Specter Ops, Inc.
//
// Licensed under the Apache License, Version 2.0
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// SPDX-License-Identifier: Apache-2.0

package ad

import (
	"context"
	"fmt"
	"log/slog"
	"sync"

	"github.com/specterops/bloodhound/analysis"
	"github.com/specterops/bloodhound/analysis/impact"
	"github.com/specterops/bloodhound/dawgs/cardinality"
	"github.com/specterops/bloodhound/dawgs/graph"
	"github.com/specterops/bloodhound/dawgs/ops"
	"github.com/specterops/bloodhound/dawgs/query"
	"github.com/specterops/bloodhound/dawgs/traversal"
	"github.com/specterops/bloodhound/dawgs/util/channels"
	"github.com/specterops/bloodhound/graphschema/ad"
)

func PostADCSESC4(ctx context.Context, tx graph.Transaction, outC chan<- analysis.CreatePostRelationshipJob, groupExpansions impact.PathAggregator, enterpriseCA *graph.Node, targetDomains *graph.NodeSet, cache ADCSCache) error {
	// 1.
	principals := cardinality.NewBitmap64()
	publishedTemplates := cache.GetPublishedTemplateCache(enterpriseCA.ID)

	// 2. iterate certtemplates that have an outbound `PublishedTo` edge to eca
	for _, certTemplate := range publishedTemplates {
		if principalsWithGenericWrite, err := FetchPrincipalsWithGenericWriteOnCertTemplate(tx, certTemplate); err != nil {
			slog.WarnContext(ctx, fmt.Sprintf("Error fetching principals with %s on cert template: %v", ad.GenericWrite, err))
		} else if principalsWithEnrollOrAllExtendedRights, err := FetchPrincipalsWithEnrollOrAllExtendedRightsOnCertTemplate(tx, certTemplate); err != nil {
			slog.WarnContext(ctx, fmt.Sprintf("Error fetching principals with %s or %s on cert template: %v", ad.Enroll, ad.AllExtendedRights, err))
		} else if principalsWithPKINameFlag, err := FetchPrincipalsWithWritePKINameFlagOnCertTemplate(tx, certTemplate); err != nil {
			slog.WarnContext(ctx, fmt.Sprintf("Error fetching principals with %s on cert template: %v", ad.WritePKINameFlag, err))
		} else if principalsWithPKIEnrollmentFlag, err := FetchPrincipalsWithWritePKIEnrollmentFlagOnCertTemplate(tx, certTemplate); err != nil {
			slog.WarnContext(ctx, fmt.Sprintf("Error fetching principals with %s on cert template: %v", ad.WritePKIEnrollmentFlag, err))
		} else if enrolleeSuppliesSubject, err := certTemplate.Properties.Get(string(ad.EnrolleeSuppliesSubject)).Bool(); err != nil {
			slog.WarnContext(ctx, fmt.Sprintf("Error fetching %s property on cert template: %v", ad.EnrolleeSuppliesSubject, err))
		} else if requiresManagerApproval, err := certTemplate.Properties.Get(string(ad.RequiresManagerApproval)).Bool(); err != nil {
			slog.WarnContext(ctx, fmt.Sprintf("Error fetching %s property on cert template: %v", ad.RequiresManagerApproval, err))
		} else {

			var (
				enterpriseCAEnrollers   = cache.GetEnterpriseCAEnrollers(enterpriseCA.ID)
				certTemplateControllers = cache.GetCertTemplateControllers(certTemplate.ID)
			)

			// 2a. principals that control the cert template
			principals.Or(
				CalculateCrossProductNodeSets(tx,
					groupExpansions,
					enterpriseCAEnrollers,
					certTemplateControllers,
				))

			// 2b. principals with `Enroll/AllExtendedRights` + `Generic Write` combination on the cert template
			principals.Or(
				CalculateCrossProductNodeSets(tx,
					groupExpansions,
					enterpriseCAEnrollers,
					principalsWithGenericWrite.Slice(),
					principalsWithEnrollOrAllExtendedRights.Slice(),
				),
			)

			// 2c. kick out early if cert template does meet conditions for ESC4
			if valid, err := isCertTemplateValidForESC4(certTemplate); err != nil {
				slog.WarnContext(ctx, fmt.Sprintf("Error validating cert template %d: %v", certTemplate.ID, err))
				continue
			} else if !valid {
				continue
			}

			// 2d. principals with `Enroll/AllExtendedRights` + `WritePKINameFlag` + `WritePKIEnrollmentFlag` on the cert template
			principals.Or(CalculateCrossProductNodeSets(tx,
				groupExpansions,
				enterpriseCAEnrollers,
				principalsWithEnrollOrAllExtendedRights.Slice(),
				principalsWithPKINameFlag.Slice(),
				principalsWithPKIEnrollmentFlag.Slice(),
			))

			// 2e.
			if enrolleeSuppliesSubject {
				principals.Or(
					CalculateCrossProductNodeSets(tx,
						groupExpansions,
						enterpriseCAEnrollers,
						principalsWithEnrollOrAllExtendedRights.Slice(),
						principalsWithPKIEnrollmentFlag.Slice(),
					),
				)
			}

			// 2f.
			if !requiresManagerApproval {
				principals.Or(
					CalculateCrossProductNodeSets(tx,
						groupExpansions,
						enterpriseCAEnrollers,
						principalsWithEnrollOrAllExtendedRights.Slice(),
						principalsWithPKINameFlag.Slice(),
					),
				)
			}
		}
	}

	principals.Each(func(value uint64) bool {
		for _, domain := range targetDomains.Slice() {
			channels.Submit(ctx, outC, analysis.CreatePostRelationshipJob{
				FromID: graph.ID(value),
				ToID:   domain.ID,
				Kind:   ad.ADCSESC4,
			})
		}
		return true
	})

	return nil
}

func isCertTemplateValidForESC4(ct *graph.Node) (bool, error) {
	if authenticationEnabled, err := ct.Properties.Get(ad.AuthenticationEnabled.String()).Bool(); err != nil {
		return false, err
	} else if !authenticationEnabled {
		return false, nil
	} else if schemaVersion, err := ct.Properties.Get(ad.SchemaVersion.String()).Float64(); err != nil {
		return false, err
	} else if authorizedSignatures, err := ct.Properties.Get(ad.AuthorizedSignatures.String()).Float64(); err != nil {
		return false, err
	} else if schemaVersion > 1 && authorizedSignatures > 0 {
		return false, nil
	} else {
		return true, nil
	}
}

func FetchPrincipalsWithGenericWriteOnCertTemplate(tx graph.Transaction, certTemplate *graph.Node) (graph.NodeSet, error) {
	if nodes, err := ops.FetchStartNodes(tx.Relationships().Filterf(
		func() graph.Criteria {
			return query.And(
				query.Equals(query.EndID(), certTemplate.ID),
				query.Kind(query.Relationship(), ad.GenericWrite),
			)
		},
	)); err != nil {
		return nil, err
	} else {
		return nodes, nil
	}
}

func FetchPrincipalsWithEnrollOrAllExtendedRightsOnCertTemplate(tx graph.Transaction, certTemplate *graph.Node) (graph.NodeSet, error) {
	if nodes, err := ops.FetchStartNodes(
		tx.Relationships().Filterf(
			func() graph.Criteria {
				return query.And(
					query.Equals(query.EndID(), certTemplate.ID),
					query.Or(
						query.Kind(query.Relationship(), ad.Enroll),
						query.Kind(query.Relationship(), ad.AllExtendedRights),
					),
				)
			},
		)); err != nil {
		return nil, err
	} else {
		return nodes, nil
	}
}

func FetchPrincipalsWithWritePKINameFlagOnCertTemplate(tx graph.Transaction, certTemplate *graph.Node) (graph.NodeSet, error) {
	if nodes, err := ops.FetchStartNodes(
		tx.Relationships().Filterf(
			func() graph.Criteria {
				return query.And(
					query.Equals(query.EndID(), certTemplate.ID),
					query.Kind(query.Relationship(), ad.WritePKINameFlag),
				)
			},
		)); err != nil {
		return nil, err
	} else {
		return nodes, nil
	}
}

func FetchPrincipalsWithWritePKIEnrollmentFlagOnCertTemplate(tx graph.Transaction, certTemplate *graph.Node) (graph.NodeSet, error) {
	if nodes, err := ops.FetchStartNodes(
		tx.Relationships().Filterf(
			func() graph.Criteria {
				return query.And(
					query.Equals(query.EndID(), certTemplate.ID),
					query.Kind(query.Relationship(), ad.WritePKIEnrollmentFlag),
				)
			},
		)); err != nil {
		return nil, err
	} else {
		return nodes, nil
	}
}

// composition: p1
func findPathsToDomainThroughCertTemplateWithGenericAll(
	ctx context.Context,
	db graph.Database,
	startNodes graph.NodeSet,
	domainID graph.ID,
	enrollAndNTAuthECAs cardinality.Duplex[uint64],
) (map[graph.ID][]*graph.PathSegment, cardinality.Duplex[uint64], error) {

	var (
		traversalInst = traversal.New(db, analysis.MaximumDatabaseParallelWorkers)
		lock          = &sync.Mutex{}

		certTemplateSegments = map[graph.ID][]*graph.PathSegment{}
		certTemplates        = cardinality.NewBitmap64()
	)

	// p1: use the enterpriseCA nodes to gather the set of cert templates with an inbound `GenericAll`
	for _, n := range startNodes.Slice() {
		if err := traversalInst.BreadthFirst(ctx, traversal.Plan{
			Root: n,
			Driver: certTemplateWithPrivelegesToDomainTraversal(
				[]graph.Kind{ad.GenericAll, ad.Owns, ad.WriteOwner, ad.WriteDACL},
				domainID,
				enrollAndNTAuthECAs,
			).Do(
				func(terminal *graph.PathSegment) error {

					certTemplate := terminal.Search(
						func(nextSegment *graph.PathSegment) bool {
							return nextSegment.Node.Kinds.ContainsOneOf(ad.CertTemplate)
						},
					)

					lock.Lock()

					certTemplateSegments[certTemplate.ID] = append(certTemplateSegments[certTemplate.ID], terminal)
					certTemplates.Add(certTemplate.ID.Uint64())

					lock.Unlock()

					return nil
				}),
		}); err != nil {
			return certTemplateSegments, certTemplates, err
		}
	}

	return certTemplateSegments, certTemplates, nil
}

// composition: p3 + p4
func findPathsToDomainThroughCertTemplateWithGenericWrite(
	ctx context.Context,
	db graph.Database,
	startNodes graph.NodeSet,
	domainID graph.ID,
	enrollAndNTAuthECAs cardinality.Duplex[uint64],
) (map[graph.ID][]*graph.PathSegment, cardinality.Duplex[uint64], error) {

	var (
		inboundEdgeToCertTemplate = ad.GenericWrite
		criteriaForCertTemplate   graph.Criteria // there aren't any!
	)

	return traversalToDomainThroughCertTemplate(ctx, db, startNodes, domainID, enrollAndNTAuthECAs, inboundEdgeToCertTemplate, criteriaForCertTemplate)
}

// composition: p6 + p7
func findPathsToDomainThroughCertTemplateWithWritePKINameFlag(
	ctx context.Context,
	db graph.Database,
	startNodes graph.NodeSet,
	domainID graph.ID,
	enrollAndNTAuthECAs cardinality.Duplex[uint64],
) (map[graph.ID][]*graph.PathSegment, cardinality.Duplex[uint64], error) {

	var (
		inboundEdgeToCertTemplate = ad.WritePKINameFlag
		criteriaForCertTemplate   = query.And(
			query.Equals(query.EndProperty(ad.RequiresManagerApproval.String()), false),
			query.Equals(query.EndProperty(ad.AuthenticationEnabled.String()), true),
			query.Or(
				query.Equals(query.EndProperty(ad.AuthorizedSignatures.String()), 0),
				query.Equals(query.EndProperty(ad.SchemaVersion.String()), 1),
			),
		)
	)

	return traversalToDomainThroughCertTemplate(ctx, db, startNodes, domainID, enrollAndNTAuthECAs, inboundEdgeToCertTemplate, criteriaForCertTemplate)

}

// composition: p9 + p10
func findPathsToDomainThroughCertTemplateWithWritePKIEnrollmentFlag(
	ctx context.Context,
	db graph.Database,
	startNodes graph.NodeSet,
	domainID graph.ID,
	enrollAndNTAuthECAs cardinality.Duplex[uint64],
) (map[graph.ID][]*graph.PathSegment, cardinality.Duplex[uint64], error) {

	var (
		inboundEdgeToCertTemplate = ad.WritePKIEnrollmentFlag
		criteriaForCertTemplate   = query.And(
			query.Equals(query.EndProperty(ad.EnrolleeSuppliesSubject.String()), true),
			query.Equals(query.EndProperty(ad.AuthenticationEnabled.String()), true),
			query.Or(
				query.Equals(query.EndProperty(ad.AuthorizedSignatures.String()), 0),
				query.Equals(query.EndProperty(ad.SchemaVersion.String()), 1),
			),
		)
	)

	return traversalToDomainThroughCertTemplate(ctx, db, startNodes, domainID, enrollAndNTAuthECAs, inboundEdgeToCertTemplate, criteriaForCertTemplate)

}

func traversalToDomainThroughCertTemplate(
	ctx context.Context,
	db graph.Database,
	startNodes graph.NodeSet,
	domainID graph.ID,
	enrollAndNTAuthECAs cardinality.Duplex[uint64],
	inboundEdgeToCertTemplate graph.Kind,
	criteriaForCertTemplate graph.Criteria,
) (map[graph.ID][]*graph.PathSegment, cardinality.Duplex[uint64], error) {

	var (
		traversalInst = traversal.New(db, analysis.MaximumDatabaseParallelWorkers)
		lock          = &sync.Mutex{}

		certTemplateSegments = map[graph.ID][]*graph.PathSegment{}
		certTemplates        = cardinality.NewBitmap64()
	)

	for _, n := range startNodes.Slice() {
		if err := traversalInst.BreadthFirst(ctx, traversal.Plan{
			Root: n,
			Driver: certTemplateWithPrivelegesToDomainTraversal([]graph.Kind{inboundEdgeToCertTemplate}, domainID, enrollAndNTAuthECAs).Do(
				func(terminal *graph.PathSegment) error {

					certTemplate := terminal.Search(
						func(nextSegment *graph.PathSegment) bool {
							return nextSegment.Node.Kinds.ContainsOneOf(ad.CertTemplate)
						},
					)

					lock.Lock()
					certTemplateSegments[certTemplate.ID] = append(certTemplateSegments[certTemplate.ID], terminal)
					certTemplates.Add(certTemplate.ID.Uint64())
					lock.Unlock()

					return nil
				}),
		}); err != nil {
			return certTemplateSegments, certTemplates, err
		}
	}

	for _, n := range startNodes.Slice() {
		if err := traversalInst.BreadthFirst(ctx, traversal.Plan{
			Root: n,
			Driver: certTemplateWithEnrollmentRightsTraversal(certTemplates, criteriaForCertTemplate).Do(
				func(terminal *graph.PathSegment) error {

					certTemplate := terminal.Search(
						func(nextSegment *graph.PathSegment) bool {
							return nextSegment.Node.Kinds.ContainsOneOf(ad.CertTemplate)
						},
					)

					lock.Lock()
					certTemplateSegments[certTemplate.ID] = append(certTemplateSegments[certTemplate.ID], terminal)
					lock.Unlock()

					return nil
				}),
		}); err != nil {
			return certTemplateSegments, certTemplates, err
		}
	}

	return certTemplateSegments, certTemplates, nil
}

// composition: p12, p13, p14
func findPathsToDomainThroughCertTemplateWithBothPKIFlags(
	ctx context.Context,
	db graph.Database,
	startNodes graph.NodeSet,
	domainID graph.ID,
	enrollAndNTAuthECAs cardinality.Duplex[uint64],
) (map[graph.ID][]*graph.PathSegment, cardinality.Duplex[uint64], error) {

	var (
		traversalInst = traversal.New(db, analysis.MaximumDatabaseParallelWorkers)
		lock          = &sync.Mutex{}

		certTemplateSegments = map[graph.ID][]*graph.PathSegment{}
		certTemplates        = cardinality.NewBitmap64()
	)

	// p9: use the enterpriseCA nodes to gather the set of cert templates with an inbound `WritePKIEnrollmentFlag`
	for _, n := range startNodes.Slice() {
		if err := traversalInst.BreadthFirst(ctx, traversal.Plan{
			Root: n,
			Driver: certTemplateWithPrivelegesToDomainTraversal([]graph.Kind{ad.WritePKIEnrollmentFlag}, domainID, enrollAndNTAuthECAs).Do(
				func(terminal *graph.PathSegment) error {

					certTemplate := terminal.Search(
						func(nextSegment *graph.PathSegment) bool {
							return nextSegment.Node.Kinds.ContainsOneOf(ad.CertTemplate)
						},
					)

					lock.Lock()
					certTemplateSegments[certTemplate.ID] = append(certTemplateSegments[certTemplate.ID], terminal)
					certTemplates.Add(certTemplate.ID.Uint64())
					lock.Unlock()

					return nil
				}),
		}); err != nil {
			return certTemplateSegments, certTemplates, err
		}
	}

	// p10: (reuse p4 logic): find cert templates that have an inbound `Enroll` OR `AllExtendedRights` edge
	for _, n := range startNodes.Slice() {
		if err := traversalInst.BreadthFirst(ctx, traversal.Plan{
			Root: n,
			Driver: certTemplateWithEnrollmentRightsTraversal(certTemplates, nil).Do(
				func(terminal *graph.PathSegment) error {

					certTemplate := terminal.Search(
						func(nextSegment *graph.PathSegment) bool {
							return nextSegment.Node.Kinds.ContainsOneOf(ad.CertTemplate)
						},
					)

					lock.Lock()
					certTemplateSegments[certTemplate.ID] = append(certTemplateSegments[certTemplate.ID], terminal)
					lock.Unlock()

					return nil
				}),
		}); err != nil {
			return certTemplateSegments, certTemplates, err
		}
	}

	// p14: find cert templates with valid combination of properties that has an inbound `WritePKIName` edge
	for _, n := range startNodes.Slice() {
		if err := traversalInst.BreadthFirst(ctx, traversal.Plan{
			Root: n,
			Driver: certTemplateWithPKINameFlagTraversal(certTemplates).Do(
				func(terminal *graph.PathSegment) error {

					certTemplate := terminal.Search(
						func(nextSegment *graph.PathSegment) bool {
							return nextSegment.Node.Kinds.ContainsOneOf(ad.CertTemplate)
						},
					)

					lock.Lock()
					certTemplateSegments[certTemplate.ID] = append(certTemplateSegments[certTemplate.ID], terminal)
					lock.Unlock()

					return nil
				}),
		}); err != nil {
			return certTemplateSegments, certTemplates, err
		}
	}

	return certTemplateSegments, certTemplates, nil
}

// composition: p2, p5, p8, p11, p15
func findPathToDomainThroughEnterpriseCAsTrustedForNTAuth(
	ctx context.Context,
	db graph.Database,
	startNodes graph.NodeSet,
	domainID graph.ID,
) (
	map[graph.ID][]*graph.PathSegment,
	cardinality.Duplex[uint64],
	error,
) {
	var (
		traversalInst = traversal.New(db, analysis.MaximumDatabaseParallelWorkers)
		lock          = &sync.Mutex{}

		enrollAndNTAuthECASegments = map[graph.ID][]*graph.PathSegment{}
		enrollAndNTAuthECAs        = cardinality.NewBitmap64()
	)

	for _, n := range startNodes.Slice() {
		if err := traversalInst.BreadthFirst(ctx,
			traversal.Plan{
				Root: n,
				Driver: ntAuthStoreToDomainTraversal(domainID).Do(
					func(terminal *graph.PathSegment) error {
						enterpriseCA := terminal.Search(
							func(nextSegment *graph.PathSegment) bool {
								return nextSegment.Node.Kinds.ContainsOneOf(ad.EnterpriseCA)
							})

						lock.Lock()
						enrollAndNTAuthECAs.Add(enterpriseCA.ID.Uint64())
						enrollAndNTAuthECASegments[enterpriseCA.ID] = append(enrollAndNTAuthECASegments[enterpriseCA.ID], terminal)
						lock.Unlock()

						return nil
					}),
			},
		); err != nil {
			return enrollAndNTAuthECASegments, enrollAndNTAuthECAs, err
		}
	}
	return enrollAndNTAuthECASegments, enrollAndNTAuthECAs, nil

}

func GetADCSESC4EdgeComposition(ctx context.Context, db graph.Database, edge *graph.Relationship) (graph.PathSet, error) {
	/*
		// 2a
		MATCH p1 = (n1)-[:MemberOf*0..]->()-[:GenericAll|Owns|WriteOwner|WriteDacl]->(ct)-[:PublishedTo]->(ca)-[:IssuedSignedBy|EnterpriseCAFor|RootCAFor*1..]->(d)
		MATCH p2 = (n1)-[:MemberOf*0..]->()-[:Enroll]->(ca)-[:TrustedForNTAuth]->(nt)-[:NTAuthStoreFor]->(d)

		// 2b
		MATCH p3 = (n2)-[:MemberOf*0..]->()-[:GenericWrite]->(ct2)-[:PublishedTo]->(ca2)-[:IssuedSignedBy|EnterpriseCAFor|RootCAFor*1..]->(d)
		MATCH p4 = (n2)-[:MemberOf*0..]->()-[:Enroll|AllExtendedRights]->(ct2)
		MATCH p5 = (n2)-[:MemberOf*0..]->()-[:Enroll]->(ca2)-[:TrustedForNTAuth]->(nt)-[:NTAuthStoreFor]->(d)

		// 2d
		MATCH p6 = (n3)-[:MemberOf*0..]->()-[:WritePKINameFlag]->(ct3)-[:PublishedTo]->(ca3)-[:IssuedSignedBy|EnterpriseCAFor|RootCAFor*1..]->(d)
		MATCH p7 = (n3)-[:MemberOf*0..]->()-[:Enroll|AllExtendedRights]->(ct3)
		WHERE ct3.requiresmanagerapproval = false
		  AND ct3.authenticationenabled = true
		  AND (
		    ct3.authorizedsignatures = 0 OR ct3.schemaversion = 1
		  )
		MATCH p8 = (n3)-[:MemberOf*0..]->()-[:Enroll]->(ca3)-[:TrustedForNTAuth]->(nt)-[:NTAuthStoreFor]->(d)

		// 2e
		MATCH p9 = (n4)-[:MemberOf*0..]->()-[:WritePKIEnrollmentFlag]->(ct4)-[:PublishedTo]->(ca4)-[:IssuedSignedBy|EnterpriseCAFor|RootCAFor*1..]->(d)
		MATCH p10 = (n4)-[:MemberOf*0..]->()-[:Enroll|AllExtendedRights]->(ct4)
		WHERE ct4.enrolleesuppliessubject = true
		  AND ct4.authenticationenabled = true
		  AND (
		    ct4.authorizedsignatures = 0 OR ct4.schemaversion = 1
		  )
		MATCH p11 = (n4)-[:MemberOf*0..]->()-[:Enroll]->(ca4)-[:TrustedForNTAuth]->(nt)-[:NTAuthStoreFor]->(d)

		// 2f
		MATCH p12 = (n5)-[:MemberOf*0..]->()-[:WritePKIEnrollmentFlag]->(ct5)-[:PublishedTo]->(ca5)-[:IssuedSignedBy|EnterpriseCAFor|RootCAFor*1..]->(d)
		MATCH p13 = (n5)-[:MemberOf*0..]->()-[:Enroll|AllExtendedRights]->(ct5)
		MATCH p14 = (n5)-[:MemberOf*0..]->()-[:WritePKINameFlag]->(ct5)
		WHERE ct5.authenticationenabled = true
		  AND (
		    ct5.authorizedsignatures = 0 OR ct5.schemaversion = 1
		  )
		MATCH p15 = (n5)-[:MemberOf*0..]->()-[:Enroll]->(ca5)-[:TrustedForNTAuth]->(nt)-[:NTAuthStoreFor]->(d)


		RETURN p1,p2,p3,p4,p5,p6,p7,p8,p9,p10,p11,p12,p13,p14,p15
	*/

	var (
		startNode  *graph.Node
		startNodes = graph.NodeSet{}

		enrollAndNTAuthECAs cardinality.Duplex[uint64]
		domainID            = edge.EndID
		paths               = graph.PathSet{}

		enrollAndNTAuthECASegments = map[graph.ID][]*graph.PathSegment{}
		finalECAs                  = cardinality.NewBitmap64()
	)

	if err := db.ReadTransaction(ctx, func(tx graph.Transaction) error {
		var err error
		if startNode, err = ops.FetchNode(tx, edge.StartID); err != nil {
			return err
		} else {
			return nil
		}
	}); err != nil {
		return nil, err
	}

	// Add startnode, Auth. Users, and Everyone to start nodes
	if err := db.ReadTransaction(ctx, func(tx graph.Transaction) error {
		if nodeSet, err := FetchAuthUsersAndEveryoneGroups(tx); err != nil {
			return err
		} else {
			startNodes.AddSet(nodeSet)
			return nil
		}
	}); err != nil {
		return nil, err
	}
	startNodes.Add(startNode)

	// p2, p5, p8, p11, p15: these pattern parts use same logic. find the paths to domain like: (eca) -> (nt auth store)-> (domain)
	if paths, ecaIDs, err := findPathToDomainThroughEnterpriseCAsTrustedForNTAuth(ctx, db, startNodes, domainID); err != nil {
		return nil, err
	} else {
		enrollAndNTAuthECASegments = paths
		enrollAndNTAuthECAs = ecaIDs
	}

	// p1, p2
	if pathsToDomain, certTemplateIDs, err := findPathsToDomainThroughCertTemplateWithGenericAll(ctx, db, startNodes, domainID, enrollAndNTAuthECAs); err != nil {
		return nil, err
	} else {
		certTemplateIDs.Each(
			func(value uint64) bool {
				// add the paths which satisfy p1-p2 requirements
				for _, segment := range pathsToDomain[graph.ID(value)] {
					paths.AddPath(segment.Path())

					// add the ECA where the template is published (first ECA in the path in case of multi-tier hierarchy) to final list of ECAs
					segment.Path().Walk(func(start, end *graph.Node, relationship *graph.Relationship) bool {
						if end.Kinds.ContainsOneOf(ad.EnterpriseCA) {
							finalECAs.Add(end.ID.Uint64())
							return false
						}
						return true
					})
				}

				return true
			},
		)
	}

	// p3, p4, p5
	if pathsToDomain, certTemplateIDs, err := findPathsToDomainThroughCertTemplateWithGenericWrite(ctx, db, startNodes, domainID, enrollAndNTAuthECAs); err != nil {
		return nil, err
	} else {
		certTemplateIDs.Each(
			func(value uint64) bool {
				// add the paths which satisfy p3, p4, and p5 requirements
				for _, segment := range pathsToDomain[graph.ID(value)] {
					paths.AddPath(segment.Path())

					// add the ECA where the template is published (first ECA in the path in case of multi-tier hierarchy) to final list of ECAs
					segment.Path().Walk(func(start, end *graph.Node, relationship *graph.Relationship) bool {
						if end.Kinds.ContainsOneOf(ad.EnterpriseCA) {
							finalECAs.Add(end.ID.Uint64())
							return false
						}
						return true
					})
				}

				return true
			},
		)
	}

	// p6, p7, p8
	if pathsToDomain, certTemplateIDs, err := findPathsToDomainThroughCertTemplateWithWritePKINameFlag(ctx, db, startNodes, domainID, enrollAndNTAuthECAs); err != nil {
		return nil, err
	} else {
		certTemplateIDs.Each(
			func(value uint64) bool {
				// add the paths which satisfy p6, p7, and p8 requirements
				for _, segment := range pathsToDomain[graph.ID(value)] {
					paths.AddPath(segment.Path())

					// add the ECA where the template is published (first ECA in the path in case of multi-tier hierarchy) to final list of ECAs
					segment.Path().Walk(func(start, end *graph.Node, relationship *graph.Relationship) bool {
						if end.Kinds.ContainsOneOf(ad.EnterpriseCA) {
							finalECAs.Add(end.ID.Uint64())
							return false
						}
						return true
					})
				}

				return true
			},
		)
	}

	// p9, p10, p11
	if pathsToDomain, certTemplateIDs, err := findPathsToDomainThroughCertTemplateWithWritePKIEnrollmentFlag(ctx, db, startNodes, domainID, enrollAndNTAuthECAs); err != nil {
		return nil, err
	} else {
		certTemplateIDs.Each(
			func(value uint64) bool {
				// add the paths which satisfy p9, p10, and p11 requirements
				for _, segment := range pathsToDomain[graph.ID(value)] {
					paths.AddPath(segment.Path())

					// add the ECA where the template is published (first ECA in the path in case of multi-tier hierarchy) to final list of ECAs
					segment.Path().Walk(func(start, end *graph.Node, relationship *graph.Relationship) bool {
						if end.Kinds.ContainsOneOf(ad.EnterpriseCA) {
							finalECAs.Add(end.ID.Uint64())
							return false
						}
						return true
					})
				}

				return true
			},
		)
	}

	// p12, p13, p14, p15
	if pathsToDomain, certTemplateIDs, err := findPathsToDomainThroughCertTemplateWithBothPKIFlags(ctx, db, startNodes, domainID, enrollAndNTAuthECAs); err != nil {
		return nil, err
	} else {
		certTemplateIDs.Each(
			func(value uint64) bool {
				// add the paths which satisfy p12, p13, p14, p15 requirements
				for _, segment := range pathsToDomain[graph.ID(value)] {
					paths.AddPath(segment.Path())

					// add the ECA where the template is published (first ECA in the path in case of multi-tier hierarchy) to final list of ECAs
					segment.Path().Walk(func(start, end *graph.Node, relationship *graph.Relationship) bool {
						if end.Kinds.ContainsOneOf(ad.EnterpriseCA) {
							finalECAs.Add(end.ID.Uint64())
							return false
						}
						return true
					})
				}

				return true
			},
		)
	}

	// if we have found paths already, materialize the paths we found from the enterprise CAs -> NTAuthStore -> Domain
	if paths.Len() > 0 {
		finalECAs.Each(
			func(value uint64) bool {
				for _, segment := range enrollAndNTAuthECASegments[graph.ID(value)] {
					paths.AddPath(segment.Path())
				}
				return true
			})
	}

	return paths, nil
}

func ntAuthStoreToDomainTraversal(domainId graph.ID) traversal.PatternContinuation {
	return traversal.NewPattern().
		OutboundWithDepth(0, 0,
			query.And(
				query.Kind(query.Relationship(), ad.MemberOf),
				query.Kind(query.End(), ad.Group),
			)).
		Outbound(
			query.And(
				query.KindIn(query.Relationship(), ad.Enroll),
				query.KindIn(query.End(), ad.EnterpriseCA),
			)).
		Outbound(
			query.And(
				query.KindIn(query.Relationship(), ad.TrustedForNTAuth),
				query.Kind(query.End(), ad.NTAuthStore),
			)).
		Outbound(
			query.And(
				query.KindIn(query.Relationship(), ad.NTAuthStoreFor),
				query.Equals(query.EndID(), domainId),
			))
}

// This traversal goes from principal -> domain via a cert template that has an inbound edge(s) corresponding to whatever `priveleges` are provided
func certTemplateWithPrivelegesToDomainTraversal(priveleges graph.Kinds, domainID graph.ID, enrollAndNTAuthECAs cardinality.Duplex[uint64]) traversal.PatternContinuation {
	return traversal.NewPattern().
		OutboundWithDepth(0, 0,
			query.And(
				query.Kind(query.Relationship(), ad.MemberOf),
				query.Kind(query.End(), ad.Group),
			)).
		Outbound(
			query.And(
				query.KindIn(query.Relationship(), priveleges...),
				query.Kind(query.End(), ad.CertTemplate),
			)).
		Outbound(query.And(
			query.KindIn(query.Relationship(), ad.PublishedTo),
			query.InIDs(query.End(), graph.DuplexToGraphIDs(enrollAndNTAuthECAs)...),
			query.Kind(query.End(), ad.EnterpriseCA),
		)).
		OutboundWithDepth(0, 0, query.And(
			query.KindIn(query.Relationship(), ad.IssuedSignedBy, ad.EnterpriseCAFor),
			query.KindIn(query.End(), ad.EnterpriseCA, ad.AIACA),
		)).
		Outbound(query.And(
			query.KindIn(query.Relationship(), ad.IssuedSignedBy, ad.EnterpriseCAFor),
			query.Kind(query.End(), ad.RootCA),
		)).
		Outbound(
			query.And(
				query.KindIn(query.Relationship(), ad.RootCAFor),
				query.Equals(query.EndID(), domainID),
			))
}

func certTemplateWithEnrollmentRightsTraversal(certTemplates cardinality.Duplex[uint64], criteria graph.Criteria) traversal.PatternContinuation {
	// start with outbound group membership of any depth
	initialTraversal := traversal.NewPattern().
		OutboundWithDepth(0, 0,
			query.And(
				query.Kind(query.Relationship(), ad.MemberOf),
				query.Kind(query.End(), ad.Group),
			))

	if criteria == nil {
		return initialTraversal.Outbound(
			query.And(
				query.KindIn(query.Relationship(), ad.Enroll, ad.AllExtendedRights),
				query.InIDs(query.End(), graph.DuplexToGraphIDs(certTemplates)...),
			))
	} else {
		return initialTraversal.Outbound(
			query.And(
				query.KindIn(query.Relationship(), ad.Enroll, ad.AllExtendedRights),
				query.InIDs(query.End(), graph.DuplexToGraphIDs(certTemplates)...),
				criteria,
			))
	}
}

func certTemplateWithPKINameFlagTraversal(certTemplates cardinality.Duplex[uint64]) traversal.PatternContinuation {
	return traversal.NewPattern().
		OutboundWithDepth(0, 0,
			query.And(
				query.Kind(query.Relationship(), ad.MemberOf),
				query.Kind(query.End(), ad.Group),
			)).
		Outbound(
			query.And(
				query.KindIn(query.Relationship(), ad.WritePKINameFlag),
				query.InIDs(query.End(), graph.DuplexToGraphIDs(certTemplates)...),
				// ct.authenticationenabled == true
				query.Equals(query.EndProperty(ad.AuthenticationEnabled.String()), true),
				query.Or(
					// ct.authorizedsignatures == 0
					query.Equals(query.EndProperty(ad.AuthorizedSignatures.String()), 0),
					// ct.schemaversion == 1
					query.Equals(query.EndProperty(ad.SchemaVersion.String()), 1),
				),
			))
}
