/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.app.util.bin.format.dwarf.line;

public class DWARFLineProgramState {
	/**
	 * The program-counter value corresponding to a machine instruction
	 * generated by the compiler.
	 */
	public long address;

	/**
	 * An unsigned integer indicating the identity of the source file
	 * corresponding to a machine instruction.
	 */
	public int file = 1;
	/**
	 * An unsigned integer indicating a source line number. Lines are
	 * numbered beginning at 1. The compiler may emit the value 0 in cases
	 * where an instruction cannot be attributed to any source line.
	 */
	public int line = 1;
	/**
	 * An unsigned integer indicating a column number within a source line.
	 * Columns are numbered beginning at 1. The value 0 is reserved to
	 * indicate that a statement begins at the ??left edge?? of the line.
	 */
	public int column;
	/**
	 * A boolean indicating that the current instruction is the beginning of a
	 * statement.
	 */
	public boolean isStatement;
	/**
	 * A boolean indicating that the current instruction is the beginning of a
	 * basic block.
	 */
	public boolean isBasicBlock;
	/**
	 * A boolean indicating that the current address is that of the first byte after
	 * the end of a sequence of target machine instructions.
	 */
	public boolean isEndSequence;

	public boolean prologueEnd;

	public boolean epilogueBegin;

	public long isa;

	public long discriminator;

	public DWARFLineProgramState(boolean defaultIsStatement) {
		this.isStatement = defaultIsStatement;
	}

	public DWARFLineProgramState(DWARFLineProgramState other) {
		this.address = other.address;
		this.file = other.file;
		this.line = other.line;
		this.column = other.column;
		this.isStatement = other.isStatement;
		this.isBasicBlock = other.isBasicBlock;
		this.isEndSequence = other.isEndSequence;
		this.prologueEnd = other.prologueEnd;
		this.epilogueBegin = other.epilogueBegin;
		this.isa = other.isa;
		this.discriminator = other.discriminator;
	}

	public boolean isSameFileLine(DWARFLineProgramState other) {
		return file == other.file && line == other.line;
	}

	@Override
	public String toString() {
		return String.format(
			"DWARFLineProgramState [address=%s, file=%s, line=%s, column=%s, isStatement=%s, isBasicBlock=%s, isEndSequence=%s, prologueEnd=%s, epilogueBegin=%s, isa=%s, discriminator=%s]",
			address, file, line, column, isStatement, isBasicBlock, isEndSequence, prologueEnd,
			epilogueBegin, isa, discriminator);
	}

}
