#
# Copyright (C) 2019-2025 Mathieu Parent <math.parent@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import annotations

from typing import TYPE_CHECKING

from gitlabracadabra.packages.package_file import PackageFile
from gitlabracadabra.packages.pulp_manifest import PulpManifestSource

if TYPE_CHECKING:
    from pytest_mock import MockerFixture


class TestPulpManifestSource:
    """Test PulpManifestSource class."""

    def test_init(self):
        """Test __init__ method."""
        source = PulpManifestSource(
            log_prefix="log-prefix",
            url="my-url",
            package_name="pkg-name",
        )
        session = source.session
        assert source.__dict__ == {
            "session": session,
            "_log_prefix": "log-prefix",
            "_url": "my-url",
            "_package_name": "pkg-name",
            "_package_version": "0",
        }

    def test_str(self):
        """Test __str__ method."""
        assert (
            str(
                PulpManifestSource(
                    log_prefix="foo ", url="https://foobar.example.org/PULP_MANIFEST", package_name="pkg-name"
                )
            )
            == "PULP_MANIFEST repository (url=https://foobar.example.org/PULP_MANIFEST)"
        )

    def test_package_files_mocked(self, mocker: MockerFixture) -> None:
        """Test package_files method, with only mandatory arguments."""
        destination = mocker.MagicMock()

        def destination_session_request(
            *args,  # noqa: ARG001
            **kwargs,  # noqa: ARG001
        ):
            response = mocker.MagicMock()
            response.status_code = 200
            response.iter_lines.return_value = [
                "file-same,sha256-same,42",
                "file-changed,sha256-changed-before,42",
                "file-to-remove,sha256-to-remove,42",
            ]
            return response

        destination.session.request.side_effect = destination_session_request

        source = PulpManifestSource(
            log_prefix="log-prefix",
            url="https://foobar.example.org/dir/PULP_MANIFEST",
            package_name="pkg-name",
        )

        def source_session_request(
            *args,  # noqa: ARG001
            **kwargs,  # noqa: ARG001
        ):
            response = mocker.MagicMock()
            response.status_code = 200
            response.iter_lines.return_value = [
                "file-same,sha256-same,42",
                "file-changed,sha256-changed-after,42",
                "file-to-add,sha256-to-add,42",
            ]
            return response

        source.session = mocker.MagicMock()
        source.session.request.side_effect = source_session_request  # type:ignore[attr-defined]

        assert source.package_files(destination) == [
            PackageFile(
                url="https://foobar.example.org/dir/file-changed",
                package_type="generic",
                package_name="pkg-name",
                package_version="0",
                file_name="file-changed",
                metadata={"sha256_checksum": "sha256-changed-after", "size_in_bytes": "42"},
                force=True,
            ),
            PackageFile(
                url="https://foobar.example.org/dir/file-to-add",
                package_type="generic",
                package_name="pkg-name",
                package_version="0",
                file_name="file-to-add",
                metadata={"sha256_checksum": "sha256-to-add", "size_in_bytes": "42"},
                force=True,
            ),
            PackageFile(
                url="https://foobar.example.org/dir/PULP_MANIFEST",
                package_type="generic",
                package_name="pkg-name",
                package_version="0",
                file_name="PULP_MANIFEST",
                force=True,
            ),
            PackageFile(
                url="https://foobar.example.org/dir/file-changed",
                package_type="generic",
                package_name="pkg-name",
                package_version="0",
                file_name="file-changed",
                metadata={"sha256_checksum": "sha256-changed-before", "size_in_bytes": "42"},
                force=True,
                delete=True,
            ),
            PackageFile(
                url="https://foobar.example.org/dir/file-to-remove",
                package_type="generic",
                package_name="pkg-name",
                package_version="0",
                file_name="file-to-remove",
                metadata={"sha256_checksum": "sha256-to-remove", "size_in_bytes": "42"},
                force=True,
                delete=True,
            ),
            PackageFile(
                url="https://foobar.example.org/dir/PULP_MANIFEST",
                package_type="generic",
                package_name="pkg-name",
                package_version="0",
                file_name="PULP_MANIFEST",
                force=True,
                delete=True,
            ),
        ]
