/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2024 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */
/////////////////////// libXpertMass includes

#include <libXpertMass/globals.hpp>
#include <libXpertMass/Utils.hpp>
#include <libXpertMass/PolChemDef.hpp>

/////////////////////// Local includes
#include "TestUtils.hpp"


namespace MsXpS
{

namespace libXpertMassCore
{


TestUtils::TestUtils()
{
  //  Note the \" to enclose the title into \".
  m_actionFormulaStringMetAlaDipeptidylTitled =
    QString("\"%1\"%2")
      .arg(m_formulaTitle)
      .arg(m_actionFormulaStringMetAlaDipeptidyl);
}


TestUtils::TestUtils(const QString &pol_chem_def_name, int version)
{
  initializePolChemDef(pol_chem_def_name, version);
}

TestUtils::~TestUtils()
{
}

void
TestUtils::initializeXpertmassLibrary()
{
  libXpertMassCore::Utils xpertmass_utils;
  xpertmass_utils.configureDebugMessagesFormat();
}

PolChemDefSPtr
TestUtils::initializePolChemDef(const QString &pol_chem_def_name, int version)
{
  initializeXpertmassLibrary();

  if(pol_chem_def_name.isEmpty())
    return nullptr;

  m_polChemDefName         = pol_chem_def_name;
  m_polChemDefFileBaseName = QString("%1.xml").arg(m_polChemDefName);
  m_polChemDefDirName      = m_polChemDefName;

  PolChemDef pol_chem_def;

  pol_chem_def.setName(m_polChemDefName);

  QString tests_pol_chem_def_input_versioned_dir =
    QString("%1/%2/version-%3")
      .arg(m_testsInputDataDir)
      .arg("polymer-chemistry-definitions")
      .arg(version);

  pol_chem_def.setXmlDataFilePath(QString("%1/%2/%3")
                                    .arg(tests_pol_chem_def_input_versioned_dir)
                                    .arg(m_polChemDefDirName)
                                    .arg(m_polChemDefFileBaseName));

  msp_polChemDef = std::make_shared<PolChemDef>(pol_chem_def);

  m_isotopeCount = PolChemDef::loadIsotopicData(msp_polChemDef);

  qDebug() << "20241110";

  if(!PolChemDef::renderXmlPolChemDefFile(msp_polChemDef))
    {
      qFatalStream() << "Failed to render the PolChemDef.";
    }
  else
    {
      ErrorList error_list;
      msp_polChemDef->validate(&error_list);
      if(!msp_polChemDef->isValid())
        qFatalStream() << "Failed to render the PolChemDef with errors:"
                 << Utils::joinErrorList(error_list, ", ");
    }

  return msp_polChemDef;
}

QDomDocument
TestUtils::craftFormulaDomDocument(const QStringList &dom_strings)
{
  int formula_element_index = 0;
  int formula_text_index    = 1;

  QDomDocument document;

  QDomElement formula_element =
    document.createElement(dom_strings[formula_element_index]);
  QDomText formula_text =
    document.createTextNode(dom_strings[formula_text_index]);
  formula_element.appendChild(formula_text);

  document.appendChild(formula_element);

  return document;
}

QDomDocument
TestUtils::craftIonizeruleDomDocument(const QStringList &dom_strings)
{
  int ionizerule_element_index = 0;
  int formula_element_index    = 1;
  int formula_text_index       = 2;
  int charge_element_index     = 3;
  int charge_text_index        = 4;
  int level_element_index      = 5;
  int level_text_index         = 6;

  QDomDocument document;

  QDomElement ionizerule_element =
    document.createElement(dom_strings[ionizerule_element_index]);

  QDomElement formula_element =
    document.createElement(dom_strings[formula_element_index]);
  QDomText formula_text =
    document.createTextNode(dom_strings[formula_text_index]);
  formula_element.appendChild(formula_text);

  QDomElement charge_element =
    document.createElement(dom_strings[charge_element_index]);
  QDomText charge_text =
    document.createTextNode(dom_strings[charge_text_index]);
  charge_element.appendChild(charge_text);

  QDomElement level_element =
    document.createElement(dom_strings[level_element_index]);
  QDomText level_text = document.createTextNode(dom_strings[level_text_index]);
  level_element.appendChild(level_text);

  ionizerule_element.appendChild(formula_element);
  ionizerule_element.appendChild(charge_element);
  ionizerule_element.appendChild(level_element);

  document.appendChild(ionizerule_element);

  return document;
}


QDomDocument
TestUtils::craftMdfDomDocument(const QStringList &dom_strings)
{
  int mdf_element_index      = 0;
  int name_element_index     = 1;
  int name_text_index        = 2;
  int formula_element_index  = 3;
  int formula_text_index     = 4;
  int targets_element_index  = 5;
  int targets_text_index     = 6;
  int maxcount_element_index = 7;
  int maxcount_text_index    = 8;

  QDomDocument document;

  QDomElement mdf_element =
    document.createElement(dom_strings[mdf_element_index]);

  QDomElement name_element =
    document.createElement(dom_strings[name_element_index]);
  QDomText name_text = document.createTextNode(dom_strings[name_text_index]);
  name_element.appendChild(name_text);

  QDomElement formula_element =
    document.createElement(dom_strings[formula_element_index]);
  QDomText formula_text =
    document.createTextNode(dom_strings[formula_text_index]);
  formula_element.appendChild(formula_text);

  QDomElement targets_element =
    document.createElement(dom_strings[targets_element_index]);
  QDomText targets_text =
    document.createTextNode(dom_strings[targets_text_index]);
  targets_element.appendChild(targets_text);

  QDomElement maxcount_element =
    document.createElement(dom_strings[maxcount_element_index]);
  QDomText maxcount_text =
    document.createTextNode(dom_strings[maxcount_text_index]);
  maxcount_element.appendChild(maxcount_text);

  mdf_element.appendChild(name_element);
  mdf_element.appendChild(formula_element);
  mdf_element.appendChild(targets_element);
  mdf_element.appendChild(maxcount_element);

  document.appendChild(mdf_element);

  return document;
}

QDomDocument
TestUtils::craftMnmDomDocument(const QStringList &dom_strings)
{
  int mnm_element_index     = 0;
  int name_element_index    = 1;
  int name_text_index       = 2;
  int code_element_index    = 3;
  int code_text_index       = 4;
  int formula_element_index = 5;
  int formula_text_index    = 6;

  QDomDocument document;

  QDomElement mnm_element =
    document.createElement(dom_strings[mnm_element_index]);

  QDomElement name_element =
    document.createElement(dom_strings[name_element_index]);
  QDomText name_text = document.createTextNode(dom_strings[name_text_index]);
  name_element.appendChild(name_text);

  QDomElement code_element =
    document.createElement(dom_strings[code_element_index]);
  QDomText code_text = document.createTextNode(dom_strings[code_text_index]);
  code_element.appendChild(code_text);

  QDomElement formula_element =
    document.createElement(dom_strings[formula_element_index]);
  QDomText formula_text =
    document.createTextNode(dom_strings[formula_text_index]);
  formula_element.appendChild(formula_text);

  mnm_element.appendChild(name_element);
  mnm_element.appendChild(code_element);
  mnm_element.appendChild(formula_element);
  document.appendChild(mnm_element);

  return document;
}

QDomDocument
TestUtils::craftMonomerDomDocument(const QStringList &dom_strings)
{
  int monomer_element_index  = 0;
  int code_element_index     = 1;
  int code_text_index        = 2;
  int mdf_element_index      = 3;
  int name_element_index     = 4;
  int name_text_index        = 5;
  int formula_element_index  = 6;
  int formula_text_index     = 7;
  int targets_element_index  = 8;
  int targets_text_index     = 9;
  int maxcount_element_index = 10;
  int maxcount_text_index    = 11;

  QDomDocument document;

  QDomElement monomer_element =
    document.createElement(dom_strings[monomer_element_index]);

  QDomElement code_element =
    document.createElement(dom_strings[code_element_index]);
  QDomText code_text = document.createTextNode(dom_strings[code_text_index]);
  code_element.appendChild(code_text);

  QDomElement mdf_element =
    document.createElement(dom_strings[mdf_element_index]);

  QDomElement name_element =
    document.createElement(dom_strings[name_element_index]);
  QDomText name_text = document.createTextNode(dom_strings[name_text_index]);
  name_element.appendChild(name_text);

  // QString str;
  // QTextStream stream(&str);
  // QDomNode node = name_element;
  // node.save(stream, 4 /*indent*/);
  // qDebug() <<  "The name element:" << str;


  QDomElement formula_element =
    document.createElement(dom_strings[formula_element_index]);
  QDomText formula_text =
    document.createTextNode(dom_strings[formula_text_index]);
  formula_element.appendChild(formula_text);

  // QString str;
  // QTextStream stream(&str);
  // QDomNode node = formula_element;
  // node.save(stream, 4 /*indent*/);
  // qDebug() <<  "The formula element:" << str;

  QDomElement targets_element =
    document.createElement(dom_strings[targets_element_index]);
  QDomText targets_text =
    document.createTextNode(dom_strings[targets_text_index]);
  targets_element.appendChild(targets_text);

  // QString str;
  // QTextStream stream(&str);
  // QDomNode node = targets_element;
  // node.save(stream, 4 /*indent*/);
  // qDebug() <<  "The targets element:" << str;

  QDomElement maxcount_element =
    document.createElement(dom_strings[maxcount_element_index]);
  QDomText maxcount_text =
    document.createTextNode(dom_strings[maxcount_text_index]);
  maxcount_element.appendChild(maxcount_text);

  // QString str;
  // QTextStream stream(&str);
  // QDomNode node = maxcount_element;
  // node.save(stream, 4 /*indent*/);
  // qDebug() <<  "The maxcount element:" << str;


  // <monomer>
  // <code>S</code>
  // <mdf>
  // <name>Phosphorylation</name>
  // <formula>H1O3P1</formula>
  // <targets>*</targets>
  // <maxcount>1</maxcount>
  // </mdf>

  // </monomer>
  mdf_element.appendChild(name_element);
  mdf_element.appendChild(formula_element);
  mdf_element.appendChild(targets_element);
  mdf_element.appendChild(maxcount_element);

  // QString str;
  // QTextStream stream(&str);
  // QDomNode node = mdf_element;
  // node.save(stream, 4 /*indent*/);
  // qDebug() <<  "The mdf element:" << str;

  monomer_element.appendChild(code_element);
  monomer_element.appendChild(mdf_element);

  // QString str;
  // QTextStream stream(&str);
  // QDomNode node = monomer_element;
  // node.save(stream, 4 /*indent*/);
  // qDebug() <<  "The monomer element:" << str;

  document.appendChild(monomer_element);

  return document;
}


QDomDocument
TestUtils::craftClkDomDocument(const QStringList &dom_strings)
{
  int clk_element_index          = 0;
  int name_element_index         = 1;
  int name_text_index            = 2;
  int formula_element_index      = 3;
  int formula_text_index         = 4;
  int modif_name_element_index_0 = 5;
  int modif_name_text_index_0    = 6;
  int modif_name_element_index_1 = 7;
  int modif_name_text_index_1    = 8;
  int modif_name_element_index_2 = 9;
  int modif_name_text_index_2    = 10;

  QDomDocument document;

  QDomElement clk_element =
    document.createElement(dom_strings[clk_element_index]);

  QDomElement name_element =
    document.createElement(dom_strings[name_element_index]);
  QDomText name_text = document.createTextNode(dom_strings[name_text_index]);
  name_element.appendChild(name_text);

  QDomElement formula_element =
    document.createElement(dom_strings[formula_element_index]);
  QDomText formula_text =
    document.createTextNode(dom_strings[formula_text_index]);
  formula_element.appendChild(formula_text);

  QDomElement modif_element_0 =
    document.createElement(dom_strings[modif_name_element_index_0]);
  QDomText modif_name_text_0 =
    document.createTextNode(dom_strings[modif_name_text_index_0]);
  modif_element_0.appendChild(modif_name_text_0);

  QDomElement modif_element_1 =
    document.createElement(dom_strings[modif_name_element_index_1]);
  QDomText modif_name_text_1 =
    document.createTextNode(dom_strings[modif_name_text_index_1]);
  modif_element_1.appendChild(modif_name_text_1);

  QDomElement modif_element_2 =
    document.createElement(dom_strings[modif_name_element_index_2]);
  QDomText modif_name_text_2 =
    document.createTextNode(dom_strings[modif_name_text_index_2]);
  modif_element_2.appendChild(modif_name_text_2);

  clk_element.appendChild(name_element);
  clk_element.appendChild(formula_element);
  clk_element.appendChild(modif_element_0);
  clk_element.appendChild(modif_element_1);
  clk_element.appendChild(modif_element_2);

  document.appendChild(clk_element);

  return document;
}

QDomDocument
TestUtils::craftClrDomDocument(const QStringList &dom_strings)
{
  //  Totally fake clr !!!
  // <clr>
  //   <name>Homoseryl</name>
  //   <le-mnm-code>A</le-mnm-code>
  //   <le-formula>+CH3COOH</le-formula>
  //   <re-mnm-code>M</re-mnm-code>
  //   <re-formula>-CH2S+O</re-formula>
  // </clr>

  int clr_element_index         = 0;
  int name_element_index        = 1;
  int name_text_index           = 2;
  int le_mnm_code_element_index = 3;
  int le_mnm_code_text_index    = 4;
  int le_formula_element_index  = 5;
  int le_formula_text_index     = 6;
  int re_mnm_code_element_index = 7;
  int re_mnm_code_text_index    = 8;
  int re_formula_element_index  = 9;
  int re_formula_text_index     = 10;

  // QStringList dom_strings{"clr", // 0
  //                         "name", // 1
  //                         "Homoseryl", // 2
  //                         "le-mnm-code", // 3
  //                         "A", // 4
  //                         "le-formula", // 5
  //                         "+CH3COOH", // 6
  //                         "re-mnm-code", // 7
  //                         "M", // 8
  //                         "re-formula", // 9
  //                         "-CH2S+O" // 10
  //                         };

  QDomDocument document;

  QDomElement clr_element =
    document.createElement(dom_strings[clr_element_index]);

  QDomElement name_element =
    document.createElement(dom_strings[name_element_index]);
  QDomText name_text = document.createTextNode(dom_strings[name_text_index]);
  name_element.appendChild(name_text);

  clr_element.appendChild(name_element);

  if(!dom_strings[le_mnm_code_element_index].isEmpty())
    {
      QDomElement le_mnm_code_element =
        document.createElement(dom_strings[le_mnm_code_element_index]);
      QDomText le_mnm_code_text =
        document.createTextNode(dom_strings[le_mnm_code_text_index]);
      le_mnm_code_element.appendChild(le_mnm_code_text);

      clr_element.appendChild(le_mnm_code_element);

      QDomElement le_formula_element =
        document.createElement(dom_strings[le_formula_element_index]);
      QDomText le_formula_text =
        document.createTextNode(dom_strings[le_formula_text_index]);
      le_formula_element.appendChild(le_formula_text);

      clr_element.appendChild(le_formula_element);
    }

  if(!dom_strings[re_mnm_code_element_index].isEmpty())
    {
      QDomElement re_mnm_code_element =
        document.createElement(dom_strings[re_mnm_code_element_index]);
      QDomText re_mnm_code_text =
        document.createTextNode(dom_strings[re_mnm_code_text_index]);
      re_mnm_code_element.appendChild(re_mnm_code_text);

      clr_element.appendChild(re_mnm_code_element);

      QDomElement re_formula_element =
        document.createElement(dom_strings[re_formula_element_index]);
      QDomText re_formula_text =
        document.createTextNode(dom_strings[re_formula_text_index]);
      re_formula_element.appendChild(re_formula_text);

      clr_element.appendChild(re_formula_element);
    }

  document.appendChild(clr_element);

  return document;
}

QDomDocument
TestUtils::craftClsDomDocument(const QStringList &dom_strings)
{
  // FAKE cls!
  // <cls>
  //   <name>CyanogenBromide</name>
  //   <pattern>M/A</pattern>
  //   <clr>
  //     <le-mnm-code>M</le-mnm-code>
  //     <le-formula>-C1H2S1+O1</le-formula>
  //     <re-mnm-code>A</re-mnm-code>
  //     <re-formula>-CH3COOH</re-formula>
  //   </clr>
  // </cls>

  // QStringList dom_strings{"cls",             // 0
  //                         "name",            // 1
  //                         "CyanogenBromide", // 2
  //                         "pattern",         // 3
  //                         "M/A",              // 4
  //                         "clr",             // 5
  //                         "name",            // 6
  //                         "Homoseryl",       // 7
  //                         "le-mnm-code",     // 8
  //                         "M",               // 9
  //                         "le-formula",      // 10
  //                         "-C1H2S1+O1",      // 11
  //                         "re-mnm-code",     // 12
  //                         "A",               // 13
  //                         "re-formula",      // 14
  //                         "-CH3COOH"};     // 15

  int cls_element_index         = 0;
  int name_element_index        = 1;
  int name_text_index           = 2;
  int pattern_element_index     = 3;
  int pattern_text_index        = 4;
  int clr_element_index         = 5;
  int clr_name_element_index    = 6;
  int clr_name_text_index       = 7;
  int le_mnm_code_element_index = 8;
  int le_mnm_code_text_index    = 9;
  int le_formula_element_index  = 10;
  int le_formula_text_index     = 11;
  int re_mnm_code_element_index = 12;
  int re_mnm_code_text_index    = 13;
  int re_formula_element_index  = 14;
  int re_formula_text_index     = 15;

  QDomDocument document;

  QDomElement cls_element =
    document.createElement(dom_strings[cls_element_index]);

  QDomElement name_element =
    document.createElement(dom_strings[name_element_index]);
  QDomText name_text = document.createTextNode(dom_strings[name_text_index]);
  name_element.appendChild(name_text);

  cls_element.appendChild(name_element);

  QDomElement pattern_element =
    document.createElement(dom_strings[pattern_element_index]);
  QDomText pattern_text =
    document.createTextNode(dom_strings[pattern_text_index]);
  pattern_element.appendChild(pattern_text);

  cls_element.appendChild(pattern_element);

  QDomElement clr_element =
    document.createElement(dom_strings[clr_element_index]);

  QDomElement clr_name_element =
    document.createElement(dom_strings[clr_name_element_index]);
  QDomText clr_name_text =
    document.createTextNode(dom_strings[clr_name_text_index]);
  clr_name_element.appendChild(clr_name_text);

  clr_element.appendChild(clr_name_element);

  QDomElement le_mnm_code_element =
    document.createElement(dom_strings[le_mnm_code_element_index]);
  QDomText le_mnm_code_text =
    document.createTextNode(dom_strings[le_mnm_code_text_index]);
  le_mnm_code_element.appendChild(le_mnm_code_text);

  clr_element.appendChild(le_mnm_code_element);

  QDomElement le_formula_element =
    document.createElement(dom_strings[le_formula_element_index]);
  QDomText le_formula_text =
    document.createTextNode(dom_strings[le_formula_text_index]);
  le_formula_element.appendChild(le_formula_text);

  clr_element.appendChild(le_formula_element);

  QDomElement re_mnm_code_element =
    document.createElement(dom_strings[re_mnm_code_element_index]);
  QDomText re_mnm_code_text =
    document.createTextNode(dom_strings[re_mnm_code_text_index]);
  re_mnm_code_element.appendChild(re_mnm_code_text);

  clr_element.appendChild(re_mnm_code_element);

  QDomElement re_formula_element =
    document.createElement(dom_strings[re_formula_element_index]);
  QDomText re_formula_text =
    document.createTextNode(dom_strings[re_formula_text_index]);
  re_formula_element.appendChild(re_formula_text);

  clr_element.appendChild(re_formula_element);

  cls_element.appendChild(clr_element);

  document.appendChild(cls_element);

  return document;
}
QDomDocument
TestUtils::craftClaDomDocument(const QStringList &dom_strings)
{
  // FAKE cla!
  // <cla>
  //   <name>CyanogenBromide</name>
  //   <pattern>M/A</pattern>
  //   <clr>
  //     <le-mnm-code>M</le-mnm-code>
  //     <le-formula>-C1H2S1+O1</le-formula>
  //     <re-mnm-code>A</re-mnm-code>
  //     <re-formula>-CH3COOH</re-formula>
  //   </clr>
  // </cla>

  // QStringList dom_strings{"cla",             // 0
  //                         "name",            // 1
  //                         "CyanogenBromide", // 2
  //                         "pattern",         // 3
  //                         "M/A",              // 4
  //                         "clr",             // 5
  //                         "name",            // 6
  //                         "Homoseryl",       // 7
  //                         "le-mnm-code",     // 8
  //                         "M",               // 9
  //                         "le-formula",      // 10
  //                         "-C1H2S1+O1",      // 11
  //                         "re-mnm-code",     // 12
  //                         "A",               // 13
  //                         "re-formula",      // 14
  //                         "-CH3COOH"};     // 15

  int cla_element_index         = 0;
  int name_element_index        = 1;
  int name_text_index           = 2;
  int pattern_element_index     = 3;
  int pattern_text_index        = 4;
  int clr_element_index         = 5;
  int clr_name_element_index    = 6;
  int clr_name_text_index       = 7;
  int le_mnm_code_element_index = 8;
  int le_mnm_code_text_index    = 9;
  int le_formula_element_index  = 10;
  int le_formula_text_index     = 11;
  int re_mnm_code_element_index = 12;
  int re_mnm_code_text_index    = 13;
  int re_formula_element_index  = 14;
  int re_formula_text_index     = 15;

  QDomDocument document;

  QDomElement cla_element =
    document.createElement(dom_strings[cla_element_index]);

  QDomElement name_element =
    document.createElement(dom_strings[name_element_index]);
  QDomText name_text = document.createTextNode(dom_strings[name_text_index]);
  name_element.appendChild(name_text);

  cla_element.appendChild(name_element);

  QDomElement pattern_element =
    document.createElement(dom_strings[pattern_element_index]);
  QDomText pattern_text =
    document.createTextNode(dom_strings[pattern_text_index]);
  pattern_element.appendChild(pattern_text);

  cla_element.appendChild(pattern_element);

  QDomElement clr_element =
    document.createElement(dom_strings[clr_element_index]);

  QDomElement clr_name_element =
    document.createElement(dom_strings[clr_name_element_index]);
  QDomText clr_name_text =
    document.createTextNode(dom_strings[clr_name_text_index]);
  clr_name_element.appendChild(clr_name_text);

  clr_element.appendChild(clr_name_element);

  QDomElement le_mnm_code_element =
    document.createElement(dom_strings[le_mnm_code_element_index]);
  QDomText le_mnm_code_text =
    document.createTextNode(dom_strings[le_mnm_code_text_index]);
  le_mnm_code_element.appendChild(le_mnm_code_text);

  clr_element.appendChild(le_mnm_code_element);

  QDomElement le_formula_element =
    document.createElement(dom_strings[le_formula_element_index]);
  QDomText le_formula_text =
    document.createTextNode(dom_strings[le_formula_text_index]);
  le_formula_element.appendChild(le_formula_text);

  clr_element.appendChild(le_formula_element);

  QDomElement re_mnm_code_element =
    document.createElement(dom_strings[re_mnm_code_element_index]);
  QDomText re_mnm_code_text =
    document.createTextNode(dom_strings[re_mnm_code_text_index]);
  re_mnm_code_element.appendChild(re_mnm_code_text);

  clr_element.appendChild(re_mnm_code_element);

  QDomElement re_formula_element =
    document.createElement(dom_strings[re_formula_element_index]);
  QDomText re_formula_text =
    document.createTextNode(dom_strings[re_formula_text_index]);
  re_formula_element.appendChild(re_formula_text);

  clr_element.appendChild(re_formula_element);

  cla_element.appendChild(clr_element);

  document.appendChild(cla_element);

  return document;
}

QDomDocument
TestUtils::craftFgrDomDocument(const QStringList &dom_strings)
{
  int fgr_element_index          = 0;
  int name_element_index         = 1;
  int name_text_index            = 2;
  int formula_element_index      = 3;
  int formula_text_index         = 4;
  int prev_code_element_index    = 5;
  int prev_code_text_index       = 6;
  int current_code_element_index = 7;
  int current_code_text_index    = 8;
  int next_code_element_index    = 9;
  int next_code_text_index       = 10;
  int comment_element_index      = 11;
  int comment_text_index         = 12;

  // QStringList dom_strings{"fgr",             // 0
  //                         "name",
  //                         "a-fgr-2",
  //                         "formula",
  //                         "+H100",
  //                         "prev-mnm-code",
  //                         "F",
  //                         "curr-mnm-code",
  //                         "D",
  //                         "next-mnm-code",
  //                         "E",
  //                         "comment",
  //                         "thecomment"};

  QDomDocument document;

  QDomElement fgr_element =
    document.createElement(dom_strings[fgr_element_index]);

  QDomElement name_element =
    document.createElement(dom_strings[name_element_index]);
  QDomText name_text = document.createTextNode(dom_strings[name_text_index]);
  name_element.appendChild(name_text);

  fgr_element.appendChild(name_element);

  QDomElement formula_element =
    document.createElement(dom_strings[formula_element_index]);
  QDomText formula_text =
    document.createTextNode(dom_strings[formula_text_index]);
  formula_element.appendChild(formula_text);

  fgr_element.appendChild(formula_element);

  //  Totally fake fgr !!!
  // <fgr>
  // <name>a-fgr-2</name>
  // <formula>+H100</formula>
  // <prev-mnm-code>F</prev-mnm-code>
  // <curr-mnm-code>D</curr-mnm-code>
  // <next-mnm-code>E</next-mnm-code>
  // <comment>thecomment</comment>
  // </fgr>

  QDomElement prev_code_element =
    document.createElement(dom_strings[prev_code_element_index]);
  QDomText prev_code_text =
    document.createTextNode(dom_strings[prev_code_text_index]);
  prev_code_element.appendChild(prev_code_text);

  fgr_element.appendChild(prev_code_element);

  QDomElement current_code_element =
    document.createElement(dom_strings[current_code_element_index]);
  QDomText current_code_text =
    document.createTextNode(dom_strings[current_code_text_index]);
  current_code_element.appendChild(current_code_text);

  fgr_element.appendChild(current_code_element);

  QDomElement next_code_element =
    document.createElement(dom_strings[next_code_element_index]);
  QDomText next_code_text =
    document.createTextNode(dom_strings[next_code_text_index]);
  next_code_element.appendChild(next_code_text);

  fgr_element.appendChild(next_code_element);

  QDomElement comment_element =
    document.createElement(dom_strings[comment_element_index]);
  QDomText comment_text =
    document.createTextNode(dom_strings[comment_text_index]);
  comment_element.appendChild(comment_text);

  fgr_element.appendChild(comment_element);

  document.appendChild(fgr_element);

  return document;
}


QDomDocument
TestUtils::craftFgsDomDocument(const QStringList &dom_strings)
{
  qDebug() << "Size of the dom string list:" << dom_strings.size();

  int fgs_element_index = 0;
  //
  int name_element_index       = 1;
  int name_text_index          = 2;
  int frag_end_element_index   = 3;
  int frag_end_text_index      = 4;
  int formula_element_index    = 5;
  int formula_text_index       = 6;
  int side_chain_element_index = 7;
  int side_chain_text_index    = 8;
  int comment_element_index    = 9;
  int comment_text_index       = 10;
  //
  int fgr_element_index = 11;
  //
  int fgr_name_element_index    = 12;
  int fgr_name_text_index       = 13;
  int fgr_formula_element_index = 14;
  int fgr_formula_text_index    = 15;
  int prev_code_element_index   = 16;
  int prev_code_text_index      = 17;
  int curr_code_element_index   = 18;
  int curr_code_text_index      = 19;
  int next_code_element_index   = 20;
  int next_code_text_index      = 21;
  int fgr_comment_element_index = 22;
  int fgr_comment_text_index    = 23;

  // <fgs>
  //   <name>a</name>
  //   <end>LE</end>
  //   <formula>-C1O1</formula>
  //   <sidechaincontrib>0</sidechaincontrib>
  //   <comment>opt_comment</comment>
  //   <fgr>
  //     <name>one_rule</name>
  //     <formula>+H2O</formula>
  //     <prev-mnm-code>M</prev-mnm-code>
  //     <curr-mnm-code>Y</curr-mnm-code>
  //     <next-mnm-code>T</next-mnm-code>
  //     <comment>opt_comment</comment>
  //   </fgr>
  //   other fgr allowed, none possible also
  // </fgs>

  QDomDocument document;

  QDomElement fgs_element =
    document.createElement(dom_strings[fgs_element_index]);

  QDomElement name_element =
    document.createElement(dom_strings[name_element_index]);
  QDomText name_text = document.createTextNode(dom_strings[name_text_index]);
  name_element.appendChild(name_text);

  fgs_element.appendChild(name_element);

  QDomElement frag_end_element =
    document.createElement(dom_strings[frag_end_element_index]);
  QDomText frag_end_text =
    document.createTextNode(dom_strings[frag_end_text_index]);
  frag_end_element.appendChild(frag_end_text);

  fgs_element.appendChild(frag_end_element);

  QDomElement formula_element =
    document.createElement(dom_strings[formula_element_index]);
  QDomText formula_text =
    document.createTextNode(dom_strings[formula_text_index]);
  formula_element.appendChild(formula_text);

  fgs_element.appendChild(formula_element);

  QDomElement side_chain_element =
    document.createElement(dom_strings[side_chain_element_index]);
  QDomText side_chain_text =
    document.createTextNode(dom_strings[side_chain_text_index]);
  side_chain_element.appendChild(side_chain_text);

  fgs_element.appendChild(side_chain_element);

  QDomElement comment_element =
    document.createElement(dom_strings[comment_element_index]);
  QDomText comment_text =
    document.createTextNode(dom_strings[comment_text_index]);
  comment_element.appendChild(comment_text);

  fgs_element.appendChild(comment_element);

  // <fgs>
  //   <name>a</name>
  //   <end>LE</end>
  //   <formula>-C1O1</formula>
  //   <comment>opt_comment</comment>
  //   <fgr>
  //     <name>one_rule</name>
  //     <formula>+H2O</formula>
  //     <prev-mnm-code>M</prev-mnm-code>
  //     <this-mnm-code>Y</this-mnm-code>
  //     <next-mnm-code>T</next-mnm-code>
  //     <comment>opt_comment</comment>
  //   </fgr>
  //   other fgr allowed, none possible also
  // </fgs>

  QDomElement fgr_element =
    document.createElement(dom_strings[fgr_element_index]);

  // int fgr_name_element_index         = 10;
  // int fgr_name_text_index            = 11;
  // int fgr_formula_element_index      = 12;
  // int fgr_formula_text_index         = 13;
  // int prev_code_element_index = 14;
  // int prev_code_text_index    = 15;
  // int curr_code_element_index = 16;
  // int curr_code_text_index    = 17;
  // int next_code_element_index    = 18;
  // int next_code_text_index       = 19;
  // int fgr_comment_element_index      = 20;
  // int fgr_comment_text_index         = 21;

  //   <fgr>
  //     <name>one_rule</name>
  //     <formula>+H2O</formula>
  //     <prev-mnm-code>M</prev-mnm-code>
  //     <curr-mnm-code>Y</curr-mnm-code>
  //     <next-mnm-code>T</next-mnm-code>
  //     <comment>opt_comment</comment>
  //   </fgr>

  QDomElement fgr_name_element =
    document.createElement(dom_strings[fgr_name_element_index]);
  QDomText fgr_name_text =
    document.createTextNode(dom_strings[fgr_name_text_index]);
  fgr_name_element.appendChild(fgr_name_text);

  fgr_element.appendChild(fgr_name_element);

  QDomElement fgr_formula_element =
    document.createElement(dom_strings[fgr_formula_element_index]);
  QDomText fgr_formula_text =
    document.createTextNode(dom_strings[fgr_formula_text_index]);
  fgr_formula_element.appendChild(fgr_formula_text);

  fgr_element.appendChild(fgr_formula_element);

  QDomElement prev_code_element =
    document.createElement(dom_strings[prev_code_element_index]);
  QDomText prev_code_text =
    document.createTextNode(dom_strings[prev_code_text_index]);
  prev_code_element.appendChild(prev_code_text);

  fgr_element.appendChild(prev_code_element);

  QDomElement curr_code_element =
    document.createElement(dom_strings[curr_code_element_index]);
  QDomText curr_code_text =
    document.createTextNode(dom_strings[curr_code_text_index]);
  curr_code_element.appendChild(curr_code_text);

  fgr_element.appendChild(curr_code_element);

  QDomElement next_code_element =
    document.createElement(dom_strings[next_code_element_index]);
  QDomText next_code_text =
    document.createTextNode(dom_strings[next_code_text_index]);
  next_code_element.appendChild(next_code_text);

  fgr_element.appendChild(next_code_element);

  QDomElement fgr_comment_element =
    document.createElement(dom_strings[fgr_comment_element_index]);
  QDomText fgr_comment_text =
    document.createTextNode(dom_strings[fgr_comment_text_index]);
  fgr_comment_element.appendChild(fgr_comment_text);

  fgr_element.appendChild(fgr_comment_element);

  fgs_element.appendChild(fgr_element);

  document.appendChild(fgs_element);

  return document;
}


} // namespace libXpertMassCore
} // namespace MsXpS
