/* BEGIN software license
 *
 * msXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the msXpertSuite project.
 *
 * The msXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Local includes
#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/OligomerPair.hpp"


namespace MsXpS
{
namespace libXpertMassCore
{


/*!
\class MsXpS::libXpertMassCore::OligomerPair
\inmodule libXpertMassCore
\ingroup XpertMassCoreMassCalculations
\inheaderfile OligomerPair.hpp

\brief The OligomerPair class provides abstractions to work with
a pair of Oligomer instances.
*/

/*!
\variable MsXpS::libXpertMassCore::OligomerPair::m_name

\brief The name of the OligomerPair.
*/

/*!
\variable MsXpS::libXpertMassCore::OligomerPair::msp_first

\brief The first Oligomer of the pair.
*/

/*!
\variable MsXpS::libXpertMassCore::OligomerPair::msp_second

\brief The second Oligomer of the pair.
*/

/*!
\variable MsXpS::libXpertMassCore::OligomerPair::m_massType

\brief The mass type that is to be handled.
*/

/*!
\variable MsXpS::libXpertMassCore::OligomerPair::m_error

\brief The error on the mass values.
*/

/*!
\variable MsXpS::libXpertMassCore::OligomerPair::m_isMatching

\brief Tells if the OligomerPair is matching.

The concept of matching depends on the use case for this class.
*/

/*!
\typedef MsXpS::libXpertMassCore::OligomerPairSPtr
\relates OligomerPair

Synonym for std::shared_ptr<OligomerPair>.
*/

/*!
\typedef MsXpS::libXpertMassCore::OligomerPairCstSPtr
\relates OligomerPair

Synonym for std::shared_ptr<const OligomerPair>.
*/

/*!
\brief Constructs an OligomerPair.

The OligomerPair instance is constructed with these arguments:

\list

\li \a name The name of this OligomerPair

\li \a first_csp The first Oligomer

\li \a first_csp The second Oligomer

\li \a mass_type Type of mass that is being dealt with

\li \a error The error

\li \a isMatching If there is a match

\endlist

\a first_csp and \a second_csp cannot be nullptr.
*/
OligomerPair::OligomerPair(const QString &name,
                           const OligomerSPtr first_csp,
                           const OligomerSPtr second_csp,
                           Enums::MassType mass_type,
                           double error,
                           bool isMatching)
  : m_name(name),
    msp_first(first_csp),
    msp_second(second_csp),
    m_massType(mass_type),
    m_error(error),
    m_isMatching(isMatching)
{
  if(msp_first == nullptr || msp_second == nullptr)
    qFatalStream() << "Programming error. Pointer(s) cannot be nullptr.";
}


/*!
\brief Constructs the OligomerPair as a copy of \a other.
*/
OligomerPair::OligomerPair(const OligomerPair &other)
  : PropListHolder(other),
    m_name(other.m_name),
    msp_first(other.msp_first),
    msp_second(other.msp_second),
    m_massType(other.m_massType),
    m_error(other.m_error),
    m_isMatching(other.m_isMatching)
{
  if(msp_first == nullptr || msp_second == nullptr)
    qFatalStream() << "Programming error. Pointer(s) cannot be nullptr.";
}


/*!
\brief Desstructs the OligomerPair.
*/
OligomerPair::~OligomerPair()
{
}


/*!
\brief Returns the name.
*/
QString
OligomerPair::getName()
{
  return m_name;
}


/*!
\brief Returns the first Oligomer.
*/
const OligomerSPtr
OligomerPair::getFirst() const
{
  return msp_first;
}


/*!
\brief Returns the second Oligomer.
*/
const OligomerSPtr
OligomerPair::getSecond() const
{
  return msp_second;
}

/*!
\brief Sets the \a mass_type.
*/
void
OligomerPair::setMassType(Enums::MassType mass_type)
{
  m_massType = mass_type;
}

/*!
\brief Returns the mass type.
*/
Enums::MassType
OligomerPair::getMassType() const
{
  return m_massType;
}

/*!
\brief Sets the \a error.
*/
void
OligomerPair::setError(double error)
{
  m_error = error;
}

/*!
\brief Returns the error.
*/
double
OligomerPair::getError() const
{
  return m_error;
}

/*!
\brief Sets the matching to \a is_matching.
*/
void
OligomerPair::setMatching(bool is_matching)
{
  m_isMatching = is_matching;
}

/*!
\brief Returns the matching.
*/
bool
OligomerPair::isMatching() const
{
  return m_isMatching;
}

/*!
\brief Returns the mass of the first Oligomer of the type set in the member
Enums::MassType.
*/
double
OligomerPair::getFirstMass()
{
  return msp_first->getMass(m_massType);
}

/*!
\brief Returns the mass of the second Oligomer of the type set in the member
Enums::MassType.
*/
double
OligomerPair::getSecondMass()
{
  return msp_second->getMass(m_massType);
}

/*!
\brief Returns the charge of the first Oligomer as calculated in the Oligomer's
Ionizer member.

\sa Ionizer
*/
int
OligomerPair::firstCharge()
{
  return msp_first->getIonizerCstRef().charge();
}


/*!
\brief Returns the charge of the second Oligomer as calculated in the Oligomer's
Ionizer member.

\sa Ionizer
*/
int
OligomerPair::secondCharge()
{
  return msp_second->getIonizerCstRef().charge();
}


//////////////// OPERATORS /////////////////////
/*!
\brief Assigns \a other to this Oligomer instance.
*/
OligomerPair &
OligomerPair::operator=(const OligomerPair &other)
{
  if(this == &other)
    return *this;

  PropListHolder::operator=(other);

  m_name       = other.m_name;
  msp_first    = other.msp_first;
  msp_second   = other.msp_second;
  m_massType   = other.m_massType;
  m_error      = other.m_error;
  m_isMatching = other.m_isMatching;

  return *this;
}

} // namespace libXpertMassCore
} // namespace MsXpS
