% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hypothesis.R
\name{hypothesis.brmsfit}
\alias{hypothesis.brmsfit}
\alias{hypothesis}
\alias{hypothesis.default}
\title{Non-Linear Hypothesis Testing}
\usage{
\method{hypothesis}{brmsfit}(
  x,
  hypothesis,
  class = "b",
  group = "",
  scope = c("standard", "ranef", "coef"),
  alpha = 0.05,
  robust = FALSE,
  seed = NULL,
  ...
)

hypothesis(x, ...)

\method{hypothesis}{default}(x, hypothesis, alpha = 0.05, robust = FALSE, ...)
}
\arguments{
\item{x}{An \code{R} object. If it is no \code{brmsfit} object,
it must be coercible to a \code{data.frame}.
In the latter case, the variables used in the \code{hypothesis} argument
need to correspond to column names of \code{x}, while the rows
are treated as representing posterior draws of the variables.}

\item{hypothesis}{A character vector specifying one or more
non-linear hypothesis concerning parameters of the model.}

\item{class}{A string specifying the class of parameters being tested.
Default is "b" for population-level effects.
Other typical options are "sd" or "cor".
If \code{class = NULL}, all parameters can be tested
against each other, but have to be specified with their full name
(see also \code{\link[brms:draws-index-brms]{variables}})}

\item{group}{Name of a grouping factor to evaluate only
group-level effects parameters related to this grouping factor.}

\item{scope}{Indicates where to look for the variables specified in
\code{hypothesis}. If \code{"standard"}, use the full parameter names
(subject to the restriction given by \code{class} and \code{group}).
If \code{"coef"} or \code{"ranef"}, compute the hypothesis for all levels
of the grouping factor given in \code{"group"}, based on the
output of \code{\link{coef.brmsfit}} and \code{\link{ranef.brmsfit}},
respectively.}

\item{alpha}{The alpha-level of the tests (default is 0.05;
see 'Details' for more information).}

\item{robust}{If \code{FALSE} (the default) the mean is used as
the measure of central tendency and the standard deviation as
the measure of variability. If \code{TRUE}, the median and the
median absolute deviation (MAD) are applied instead.}

\item{seed}{A single numeric value passed to \code{\link{set.seed}}
to make results reproducible. This is currently only relevant for point
hypotheses that scope over at least two parameters (see Details).}

\item{...}{Currently ignored.}
}
\value{
A \code{\link{brmshypothesis}} object.
}
\description{
Perform non-linear hypothesis testing for all model parameters.
}
\details{
Among others, \code{hypothesis} computes an evidence ratio
  (\code{Evid.Ratio}) for each hypothesis. For a one-sided hypothesis, this
  is just the posterior probability (\code{Post.Prob}) under the hypothesis
  against its alternative. That is, when the hypothesis is of the form
  \code{a > b}, the evidence ratio is the ratio of the posterior probability
  of \code{a > b} and the posterior probability of \code{a < b}. In this
  example, values greater than one indicate that the evidence in favor of
  \code{a > b} is larger than evidence in favor of \code{a < b}. For a
  two-sided (point) hypothesis, the evidence ratio is a Bayes factor between
  the hypothesis and its alternative computed via the Savage-Dickey density
  ratio method. That is the posterior density at the point of interest
  divided by the prior density at that point. Values greater than one
  indicate that evidence in favor of the point hypothesis has increased after
  seeing the data. In order to calculate this Bayes factor, all parameters
  related to the hypothesis must have proper priors and argument
  \code{sample_prior} of function \code{brm} must be set to \code{"yes"}.
  Otherwise \code{Evid.Ratio} (and \code{Post.Prob}) will be \code{NA}.

  Please note that the Savage-Dickey density ratio as implemented here provides
  only a very basic test of point hypotheses. It is recommended that you use
  bridge sampling instead (via \code{\link{bayes_factor}} which relies on the
  \pkg{bridgesampling} package). When interpreting Bayes factors for point
  hypotheses, make sure that your priors are reasonable and carefully chosen,
  as the result will depend heavily on the priors. In particular, avoid using
  default priors. Additionally, note that point hypotheses that scope over more
  than one parameter (e.g., when testing equality between two parameters) involve
  random sampling of the priors over those parameters (to accommodate the
  the assumption that priors for different parameters are independent of each other).
  This introduces an element of randomness into such hypothesis tests. Consider
  repeating the test to assure results are sufficiently stable, and use the argument
  \code{seed} for reproducibility. Finally, note that, for technical reasons, we
  cannot sample from priors of certain parameters classes. Most notably, these include
  overall intercept parameters (prior class \code{"Intercept"}) as well as group-level
  coefficients.

  For one-sided hypotheses, the \code{Evid.Ratio} may sometimes be \code{0} or \code{Inf} implying very
  small or large evidence, respectively, in favor of the tested hypothesis.
  For one-sided hypotheses pairs, this basically means that all posterior
  draws are on the same side of the value dividing the two hypotheses. In
  that sense, instead of \code{0} or \code{Inf,} you may rather read it as
  \code{Evid.Ratio} smaller \code{1 / S} or greater \code{S}, respectively,
  where \code{S} denotes the number of posterior draws used in the
  computations.

  The argument \code{alpha} specifies the size of the credible interval
  (i.e., Bayesian confidence interval). For instance, if we tested a
  two-sided hypothesis and set \code{alpha = 0.05} (5\%) an, the credible
  interval will contain \code{1 - alpha = 0.95} (95\%) of the posterior
  values. Hence, \code{alpha * 100}\% of the posterior values will
  lie outside of the credible interval. Although this allows testing of
  hypotheses in a similar manner as in the frequentist null-hypothesis
  testing framework, we strongly argue against using arbitrary cutoffs (e.g.,
  \code{p < .05}) to determine the 'existence' of an effect.
}
\examples{
\dontrun{
## define priors
prior <- c(set_prior("normal(0,2)", class = "b"),
           set_prior("student_t(10,0,1)", class = "sigma"),
           set_prior("student_t(10,0,1)", class = "sd"))

## fit a linear mixed effects models
fit <- brm(time ~ age + sex + disease + (1 + age|patient),
           data = kidney, family = lognormal(),
           prior = prior, sample_prior = "yes",
           control = list(adapt_delta = 0.95))

## perform two-sided hypothesis testing
(hyp1 <- hypothesis(fit, "sexfemale = age + diseasePKD"))
plot(hyp1)
hypothesis(fit, "exp(age) - 3 = 0", alpha = 0.01)

## perform one-sided hypothesis testing
hypothesis(fit, "diseasePKD + diseaseGN - 3 < 0")

hypothesis(fit, "age < Intercept",
           class = "sd", group  = "patient")

## test the amount of random intercept variance on all variance
h <- paste("sd_patient__Intercept^2 / (sd_patient__Intercept^2 +",
           "sd_patient__age^2 + sigma^2) = 0")
(hyp2 <- hypothesis(fit, h, class = NULL))
plot(hyp2)

## test more than one hypothesis at once
h <- c("diseaseGN = diseaseAN", "2 * diseaseGN - diseasePKD = 0")
(hyp3 <- hypothesis(fit, h))
plot(hyp3, ignore_prior = TRUE)

## compute hypotheses for all levels of a grouping factor
hypothesis(fit, "age = 0", scope = "coef", group = "patient")

## use the default method
dat <- as.data.frame(fit)
str(dat)
hypothesis(dat, "b_age > 0")
}

}
\seealso{
\code{\link{brmshypothesis}}
}
