use crate::lex::lex;
use crate::types::*;
use crate::SyntaxKind;
use crate::SyntaxKind::*;
use crate::DEFAULT_VERSION;
use std::str::FromStr;

#[derive(Debug, Clone, PartialEq, Eq, Hash)]
pub struct ParseError(Vec<String>);

impl std::fmt::Display for ParseError {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        for err in &self.0 {
            writeln!(f, "{}", err)?;
        }
        Ok(())
    }
}

impl std::error::Error for ParseError {}

/// Second, implementing the `Language` trait teaches rowan to convert between
/// these two SyntaxKind types, allowing for a nicer SyntaxNode API where
/// "kinds" are values from our `enum SyntaxKind`, instead of plain u16 values.
#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord, Hash)]
enum Lang {}
impl rowan::Language for Lang {
    type Kind = SyntaxKind;
    fn kind_from_raw(raw: rowan::SyntaxKind) -> Self::Kind {
        unsafe { std::mem::transmute::<u16, SyntaxKind>(raw.0) }
    }
    fn kind_to_raw(kind: Self::Kind) -> rowan::SyntaxKind {
        kind.into()
    }
}

/// GreenNode is an immutable tree, which is cheap to change,
/// but doesn't contain offsets and parent pointers.
use rowan::GreenNode;

/// You can construct GreenNodes by hand, but a builder
/// is helpful for top-down parsers: it maintains a stack
/// of currently in-progress nodes
use rowan::GreenNodeBuilder;

/// The parse results are stored as a "green tree".
/// We'll discuss working with the results later
struct Parse {
    green_node: GreenNode,
    #[allow(unused)]
    errors: Vec<String>,
    #[allow(unused)]
    version: i32,
}

fn parse(text: &str) -> Parse {
    struct Parser {
        /// input tokens, including whitespace,
        /// in *reverse* order.
        tokens: Vec<(SyntaxKind, String)>,
        /// the in-progress tree.
        builder: GreenNodeBuilder<'static>,
        /// the list of syntax errors we've accumulated
        /// so far.
        errors: Vec<String>,
    }

    impl Parser {
        fn parse_version(&mut self) -> Option<i32> {
            let mut version = None;
            if self.tokens.last() == Some(&(KEY, "version".to_string())) {
                self.builder.start_node(VERSION.into());
                self.bump();
                self.skip_ws();
                if self.current() != Some(EQUALS) {
                    self.builder.start_node(ERROR.into());
                    self.errors.push("expected `=`".to_string());
                    self.bump();
                    self.builder.finish_node();
                } else {
                    self.bump();
                }
                if self.current() != Some(VALUE) {
                    self.builder.start_node(ERROR.into());
                    self.errors
                        .push(format!("expected value, got {:?}", self.current()));
                    self.bump();
                    self.builder.finish_node();
                } else {
                    let version_str = self.tokens.last().unwrap().1.clone();
                    match version_str.parse() {
                        Ok(v) => {
                            version = Some(v);
                            self.bump();
                        }
                        Err(_) => {
                            self.builder.start_node(ERROR.into());
                            self.errors
                                .push(format!("invalid version: {}", version_str));
                            self.bump();
                            self.builder.finish_node();
                        }
                    }
                }
                if self.current() != Some(NEWLINE) {
                    self.builder.start_node(ERROR.into());
                    self.errors.push("expected newline".to_string());
                    self.bump();
                    self.builder.finish_node();
                } else {
                    self.bump();
                }
                self.builder.finish_node();
            }
            version
        }

        fn parse_watch_entry(&mut self) -> bool {
            self.skip_ws();
            if self.current().is_none() {
                return false;
            }
            if self.current() == Some(NEWLINE) {
                self.bump();
                return false;
            }
            self.builder.start_node(ENTRY.into());
            self.parse_options_list();
            for i in 0..4 {
                if self.current() == Some(NEWLINE) {
                    break;
                }
                if self.current() == Some(CONTINUATION) {
                    self.bump();
                    self.skip_ws();
                    continue;
                }
                if self.current() != Some(VALUE) && self.current() != Some(KEY) {
                    self.builder.start_node(ERROR.into());
                    self.errors.push(format!(
                        "expected value, got {:?} (i={})",
                        self.current(),
                        i
                    ));
                    if self.current().is_some() {
                        self.bump();
                    }
                    self.builder.finish_node();
                } else {
                    // Wrap each field in its appropriate node
                    match i {
                        0 => {
                            // URL
                            self.builder.start_node(URL.into());
                            self.bump();
                            self.builder.finish_node();
                        }
                        1 => {
                            // Matching pattern
                            self.builder.start_node(MATCHING_PATTERN.into());
                            self.bump();
                            self.builder.finish_node();
                        }
                        2 => {
                            // Version policy
                            self.builder.start_node(VERSION_POLICY.into());
                            self.bump();
                            self.builder.finish_node();
                        }
                        3 => {
                            // Script
                            self.builder.start_node(SCRIPT.into());
                            self.bump();
                            self.builder.finish_node();
                        }
                        _ => {
                            self.bump();
                        }
                    }
                }
                self.skip_ws();
            }
            if self.current() != Some(NEWLINE) && self.current().is_some() {
                self.builder.start_node(ERROR.into());
                self.errors
                    .push(format!("expected newline, not {:?}", self.current()));
                if self.current().is_some() {
                    self.bump();
                }
                self.builder.finish_node();
            } else {
                self.bump();
            }
            self.builder.finish_node();
            true
        }

        fn parse_option(&mut self) -> bool {
            if self.current().is_none() {
                return false;
            }
            while self.current() == Some(CONTINUATION) {
                self.bump();
            }
            if self.current() == Some(WHITESPACE) {
                return false;
            }
            self.builder.start_node(OPTION.into());
            if self.current() != Some(KEY) {
                self.builder.start_node(ERROR.into());
                self.errors.push("expected key".to_string());
                self.bump();
                self.builder.finish_node();
            } else {
                self.bump();
            }
            if self.current() == Some(EQUALS) {
                self.bump();
                if self.current() != Some(VALUE) && self.current() != Some(KEY) {
                    self.builder.start_node(ERROR.into());
                    self.errors
                        .push(format!("expected value, got {:?}", self.current()));
                    self.bump();
                    self.builder.finish_node();
                } else {
                    self.bump();
                }
            } else if self.current() == Some(COMMA) {
            } else {
                self.builder.start_node(ERROR.into());
                self.errors.push("expected `=`".to_string());
                if self.current().is_some() {
                    self.bump();
                }
                self.builder.finish_node();
            }
            self.builder.finish_node();
            true
        }

        fn parse_options_list(&mut self) {
            self.skip_ws();
            if self.tokens.last() == Some(&(KEY, "opts".to_string()))
                || self.tokens.last() == Some(&(KEY, "options".to_string()))
            {
                self.builder.start_node(OPTS_LIST.into());
                self.bump();
                self.skip_ws();
                if self.current() != Some(EQUALS) {
                    self.builder.start_node(ERROR.into());
                    self.errors.push("expected `=`".to_string());
                    if self.current().is_some() {
                        self.bump();
                    }
                    self.builder.finish_node();
                } else {
                    self.bump();
                }
                let quoted = if self.current() == Some(QUOTE) {
                    self.bump();
                    true
                } else {
                    false
                };
                loop {
                    if quoted {
                        if self.current() == Some(QUOTE) {
                            self.bump();
                            break;
                        }
                        self.skip_ws();
                    }
                    if !self.parse_option() {
                        break;
                    }
                    if self.current() == Some(COMMA) {
                        self.bump();
                    } else if !quoted {
                        break;
                    }
                }
                self.builder.finish_node();
                self.skip_ws();
            }
        }

        fn parse(mut self) -> Parse {
            let mut version = 1;
            // Make sure that the root node covers all source
            self.builder.start_node(ROOT.into());
            if let Some(v) = self.parse_version() {
                version = v;
            }
            // TODO: use version to influence parsing
            loop {
                if !self.parse_watch_entry() {
                    break;
                }
            }
            // Don't forget to eat *trailing* whitespace
            self.skip_ws();
            // Close the root node.
            self.builder.finish_node();

            // Turn the builder into a GreenNode
            Parse {
                green_node: self.builder.finish(),
                errors: self.errors,
                version,
            }
        }
        /// Advance one token, adding it to the current branch of the tree builder.
        fn bump(&mut self) {
            let (kind, text) = self.tokens.pop().unwrap();
            self.builder.token(kind.into(), text.as_str());
        }
        /// Peek at the first unprocessed token
        fn current(&self) -> Option<SyntaxKind> {
            self.tokens.last().map(|(kind, _)| *kind)
        }
        fn skip_ws(&mut self) {
            while self.current() == Some(WHITESPACE)
                || self.current() == Some(CONTINUATION)
                || self.current() == Some(COMMENT)
            {
                self.bump()
            }
        }
    }

    let mut tokens = lex(text);
    tokens.reverse();
    Parser {
        tokens,
        builder: GreenNodeBuilder::new(),
        errors: Vec::new(),
    }
    .parse()
}

/// To work with the parse results we need a view into the
/// green tree - the Syntax tree.
/// It is also immutable, like a GreenNode,
/// but it contains parent pointers, offsets, and
/// has identity semantics.

type SyntaxNode = rowan::SyntaxNode<Lang>;
#[allow(unused)]
type SyntaxToken = rowan::SyntaxToken<Lang>;
#[allow(unused)]
type SyntaxElement = rowan::NodeOrToken<SyntaxNode, SyntaxToken>;

impl Parse {
    fn syntax(&self) -> SyntaxNode {
        SyntaxNode::new_root_mut(self.green_node.clone())
    }

    fn root(&self) -> WatchFile {
        WatchFile::cast(self.syntax()).unwrap()
    }
}

macro_rules! ast_node {
    ($ast:ident, $kind:ident) => {
        #[derive(PartialEq, Eq, Hash)]
        #[repr(transparent)]
        /// A node in the syntax tree for $ast
        pub struct $ast(SyntaxNode);
        impl $ast {
            #[allow(unused)]
            fn cast(node: SyntaxNode) -> Option<Self> {
                if node.kind() == $kind {
                    Some(Self(node))
                } else {
                    None
                }
            }
        }

        impl ToString for $ast {
            fn to_string(&self) -> String {
                self.0.text().to_string()
            }
        }
    };
}

ast_node!(WatchFile, ROOT);
ast_node!(Version, VERSION);
ast_node!(Entry, ENTRY);
ast_node!(OptionList, OPTS_LIST);
ast_node!(_Option, OPTION);
ast_node!(Url, URL);
ast_node!(MatchingPattern, MATCHING_PATTERN);
ast_node!(VersionPolicyNode, VERSION_POLICY);
ast_node!(ScriptNode, SCRIPT);

impl WatchFile {
    /// Create a new watch file with specified version
    pub fn new(version: Option<u32>) -> WatchFile {
        let mut builder = GreenNodeBuilder::new();

        builder.start_node(ROOT.into());
        if let Some(version) = version {
            builder.start_node(VERSION.into());
            builder.token(KEY.into(), "version");
            builder.token(EQUALS.into(), "=");
            builder.token(VALUE.into(), version.to_string().as_str());
            builder.token(NEWLINE.into(), "\n");
            builder.finish_node();
        }
        builder.finish_node();
        WatchFile(SyntaxNode::new_root_mut(builder.finish()))
    }

    /// Returns the version of the watch file.
    pub fn version(&self) -> u32 {
        self.0
            .children()
            .find_map(Version::cast)
            .map(|it| it.version())
            .unwrap_or(DEFAULT_VERSION)
    }

    /// Returns an iterator over all entries in the watch file.
    pub fn entries(&self) -> impl Iterator<Item = Entry> + '_ {
        self.0.children().filter_map(Entry::cast)
    }

    /// Set the version of the watch file.
    pub fn set_version(&mut self, new_version: u32) {
        // Build the new version node
        let mut builder = GreenNodeBuilder::new();
        builder.start_node(VERSION.into());
        builder.token(KEY.into(), "version");
        builder.token(EQUALS.into(), "=");
        builder.token(VALUE.into(), new_version.to_string().as_str());
        builder.token(NEWLINE.into(), "\n");
        builder.finish_node();
        let new_version_green = builder.finish();

        // Create a syntax node (splice_children will detach and reattach it)
        let new_version_node = SyntaxNode::new_root_mut(new_version_green);

        // Find existing version node if any
        let version_pos = self.0.children().position(|child| child.kind() == VERSION);

        if let Some(pos) = version_pos {
            // Replace existing version node
            self.0
                .splice_children(pos..pos + 1, vec![new_version_node.into()]);
        } else {
            // Insert version node at the beginning
            self.0.splice_children(0..0, vec![new_version_node.into()]);
        }
    }
}

impl FromStr for WatchFile {
    type Err = ParseError;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        let parsed = parse(s);
        if parsed.errors.is_empty() {
            Ok(parsed.root())
        } else {
            Err(ParseError(parsed.errors))
        }
    }
}

impl Version {
    /// Returns the version of the watch file.
    pub fn version(&self) -> u32 {
        self.0
            .children_with_tokens()
            .find_map(|it| match it {
                SyntaxElement::Token(token) => {
                    if token.kind() == VALUE {
                        Some(token.text().parse().unwrap())
                    } else {
                        None
                    }
                }
                _ => None,
            })
            .unwrap_or(DEFAULT_VERSION)
    }
}

impl Entry {
    /// List of options
    pub fn option_list(&self) -> Option<OptionList> {
        self.0.children().find_map(OptionList::cast)
    }

    /// Get the value of an option
    pub fn get_option(&self, key: &str) -> Option<String> {
        self.option_list().and_then(|ol| ol.get_option(key))
    }

    /// Check if an option is set
    pub fn has_option(&self, key: &str) -> bool {
        self.option_list().map_or(false, |ol| ol.has_option(key))
    }

    /// The name of the secondary source tarball
    pub fn component(&self) -> Option<String> {
        self.get_option("component")
    }

    /// Component type
    pub fn ctype(&self) -> Result<Option<ComponentType>, ()> {
        self.get_option("ctype").map(|s| s.parse()).transpose()
    }

    /// Compression method
    pub fn compression(&self) -> Result<Option<Compression>, ()> {
        self.get_option("compression")
            .map(|s| s.parse())
            .transpose()
    }

    /// Repack the tarball
    pub fn repack(&self) -> bool {
        self.has_option("repack")
    }

    /// Repack suffix
    pub fn repacksuffix(&self) -> Option<String> {
        self.get_option("repacksuffix")
    }

    /// Retrieve the mode of the watch file entry.
    pub fn mode(&self) -> Result<Mode, ()> {
        Ok(self
            .get_option("mode")
            .map(|s| s.parse())
            .transpose()?
            .unwrap_or_default())
    }

    /// Return the git pretty mode
    pub fn pretty(&self) -> Result<Pretty, ()> {
        Ok(self
            .get_option("pretty")
            .map(|s| s.parse())
            .transpose()?
            .unwrap_or_default())
    }

    /// Set the date string used by the pretty option to an arbitrary format as an optional
    /// opts argument when the matching-pattern is HEAD or heads/branch for git mode.
    pub fn date(&self) -> String {
        self.get_option("date")
            .unwrap_or_else(|| "%Y%m%d".to_string())
    }

    /// Return the git export mode
    pub fn gitexport(&self) -> Result<GitExport, ()> {
        Ok(self
            .get_option("gitexport")
            .map(|s| s.parse())
            .transpose()?
            .unwrap_or_default())
    }

    /// Return the git mode
    pub fn gitmode(&self) -> Result<GitMode, ()> {
        Ok(self
            .get_option("gitmode")
            .map(|s| s.parse())
            .transpose()?
            .unwrap_or_default())
    }

    /// Return the pgp mode
    pub fn pgpmode(&self) -> Result<PgpMode, ()> {
        Ok(self
            .get_option("pgpmode")
            .map(|s| s.parse())
            .transpose()?
            .unwrap_or_default())
    }

    /// Return the search mode
    pub fn searchmode(&self) -> Result<SearchMode, ()> {
        Ok(self
            .get_option("searchmode")
            .map(|s| s.parse())
            .transpose()?
            .unwrap_or_default())
    }

    /// Return the decompression mode
    pub fn decompress(&self) -> bool {
        self.has_option("decompress")
    }

    /// Whether to disable all site specific special case code such as URL director uses and page
    /// content alterations.
    pub fn bare(&self) -> bool {
        self.has_option("bare")
    }

    /// Set the user-agent string used to contact the HTTP(S) server as user-agent-string. (persistent)
    pub fn user_agent(&self) -> Option<String> {
        self.get_option("user-agent")
    }

    /// Use PASV mode for the FTP connection.
    pub fn passive(&self) -> Option<bool> {
        if self.has_option("passive") || self.has_option("pasv") {
            Some(true)
        } else if self.has_option("active") || self.has_option("nopasv") {
            Some(false)
        } else {
            None
        }
    }

    /// Add the extra options to use with the unzip command, such as -a, -aa, and -b, when executed
    /// by mk-origtargz.
    pub fn unzipoptions(&self) -> Option<String> {
        self.get_option("unzipopt")
    }

    /// Normalize the downloaded web page string.
    pub fn dversionmangle(&self) -> Option<String> {
        self.get_option("dversionmangle")
            .or_else(|| self.get_option("versionmangle"))
    }

    /// Normalize the directory path string matching the regex in a set of parentheses of
    /// http://URL as the sortable version index string.  This is used
    /// as the directory path sorting index only.
    pub fn dirversionmangle(&self) -> Option<String> {
        self.get_option("dirversionmangle")
    }

    /// Normalize the downloaded web page string.
    pub fn pagemangle(&self) -> Option<String> {
        self.get_option("pagemangle")
    }

    /// Normalize the candidate upstream version strings extracted from hrefs in the
    /// source of the web page.  This is used as the version sorting index when selecting the
    /// latest upstream version.
    pub fn uversionmangle(&self) -> Option<String> {
        self.get_option("uversionmangle")
            .or_else(|| self.get_option("versionmangle"))
    }

    /// Syntactic shorthand for uversionmangle=rules, dversionmangle=rules
    pub fn versionmangle(&self) -> Option<String> {
        self.get_option("versionmangle")
    }

    /// Convert the selected upstream tarball href string from the percent-encoded hexadecimal
    /// string to the decoded normal URL  string  for  obfuscated
    /// web sites.  Only percent-encoding is available and it is decoded with
    /// s/%([A-Fa-f\d]{2})/chr hex $1/eg.
    pub fn hrefdecode(&self) -> bool {
        self.get_option("hrefdecode").is_some()
    }

    /// Convert the selected upstream tarball href string into the accessible URL for obfuscated
    /// web sites.  This is run after hrefdecode.
    pub fn downloadurlmangle(&self) -> Option<String> {
        self.get_option("downloadurlmangle")
    }

    /// Generate the upstream tarball filename from the selected href string if matching-pattern
    /// can extract the latest upstream version <uversion> from the  selected  href  string.
    /// Otherwise, generate the upstream tarball filename from its full URL string and set the
    /// missing <uversion> from the generated upstream tarball filename.
    ///
    /// Without this option, the default upstream tarball filename is generated by taking the last
    /// component of the URL and  removing everything  after any '?' or '#'.
    pub fn filenamemangle(&self) -> Option<String> {
        self.get_option("filenamemangle")
    }

    /// Generate the candidate upstream signature file URL string from the upstream tarball URL.
    pub fn pgpsigurlmangle(&self) -> Option<String> {
        self.get_option("pgpsigurlmangle")
    }

    /// Generate the version string <oversion> of the source tarball <spkg>_<oversion>.orig.tar.gz
    /// from <uversion>.  This should be used to add a suffix such as +dfsg to a MUT package.
    pub fn oversionmangle(&self) -> Option<String> {
        self.get_option("oversionmangle")
    }

    /// Returns options set
    pub fn opts(&self) -> std::collections::HashMap<String, String> {
        let mut options = std::collections::HashMap::new();

        if let Some(ol) = self.option_list() {
            for opt in ol.children() {
                let key = opt.key();
                let value = opt.value();
                if let (Some(key), Some(value)) = (key, value) {
                    options.insert(key.to_string(), value.to_string());
                }
            }
        }

        options
    }

    fn items(&self) -> impl Iterator<Item = String> + '_ {
        self.0.children_with_tokens().filter_map(|it| match it {
            SyntaxElement::Token(token) => {
                if token.kind() == VALUE || token.kind() == KEY {
                    Some(token.text().to_string())
                } else {
                    None
                }
            }
            SyntaxElement::Node(node) => {
                // Extract values from entry field nodes
                match node.kind() {
                    URL => Url::cast(node).map(|n| n.url()),
                    MATCHING_PATTERN => MatchingPattern::cast(node).map(|n| n.pattern()),
                    VERSION_POLICY => VersionPolicyNode::cast(node).map(|n| n.policy()),
                    SCRIPT => ScriptNode::cast(node).map(|n| n.script()),
                    _ => None,
                }
            }
        })
    }

    /// Returns the URL of the entry.
    pub fn url(&self) -> String {
        self.0
            .children()
            .find_map(Url::cast)
            .map(|it| it.url())
            .unwrap_or_else(|| {
                // Fallback for entries without URL node (shouldn't happen with new parser)
                self.items().next().unwrap()
            })
    }

    /// Returns the matching pattern of the entry.
    pub fn matching_pattern(&self) -> Option<String> {
        self.0
            .children()
            .find_map(MatchingPattern::cast)
            .map(|it| it.pattern())
            .or_else(|| {
                // Fallback for entries without MATCHING_PATTERN node
                self.items().nth(1)
            })
    }

    /// Returns the version policy
    pub fn version(&self) -> Result<Option<crate::VersionPolicy>, String> {
        self.0
            .children()
            .find_map(VersionPolicyNode::cast)
            .map(|it| it.policy().parse())
            .transpose()
            .or_else(|_e| {
                // Fallback for entries without VERSION_POLICY node
                self.items().nth(2).map(|it| it.parse()).transpose()
            })
    }

    /// Returns the script of the entry.
    pub fn script(&self) -> Option<String> {
        self.0
            .children()
            .find_map(ScriptNode::cast)
            .map(|it| it.script())
            .or_else(|| {
                // Fallback for entries without SCRIPT node
                self.items().nth(3)
            })
    }

    /// Replace all substitutions and return the resulting URL.
    pub fn format_url(&self, package: impl FnOnce() -> String) -> url::Url {
        subst(self.url().as_str(), package).parse().unwrap()
    }

    /// Set the URL of the entry.
    pub fn set_url(&mut self, new_url: &str) {
        // Build the new URL node
        let mut builder = GreenNodeBuilder::new();
        builder.start_node(URL.into());
        builder.token(VALUE.into(), new_url);
        builder.finish_node();
        let new_url_green = builder.finish();

        // Create a syntax node (splice_children will detach and reattach it)
        let new_url_node = SyntaxNode::new_root_mut(new_url_green);

        // Find existing URL node position (need to use children_with_tokens for correct indexing)
        let url_pos = self
            .0
            .children_with_tokens()
            .position(|child| matches!(child, SyntaxElement::Node(node) if node.kind() == URL));

        if let Some(pos) = url_pos {
            // Replace existing URL node
            self.0
                .splice_children(pos..pos + 1, vec![new_url_node.into()]);
        }
    }

    /// Set the matching pattern of the entry.
    ///
    /// TODO: This currently only replaces an existing matching pattern.
    /// If the entry doesn't have a matching pattern, this method does nothing.
    /// Future implementation should insert the node at the correct position.
    pub fn set_matching_pattern(&mut self, new_pattern: &str) {
        // Build the new MATCHING_PATTERN node
        let mut builder = GreenNodeBuilder::new();
        builder.start_node(MATCHING_PATTERN.into());
        builder.token(VALUE.into(), new_pattern);
        builder.finish_node();
        let new_pattern_green = builder.finish();

        // Create a syntax node (splice_children will detach and reattach it)
        let new_pattern_node = SyntaxNode::new_root_mut(new_pattern_green);

        // Find existing MATCHING_PATTERN node position
        let pattern_pos = self.0.children_with_tokens().position(
            |child| matches!(child, SyntaxElement::Node(node) if node.kind() == MATCHING_PATTERN),
        );

        if let Some(pos) = pattern_pos {
            // Replace existing MATCHING_PATTERN node
            self.0
                .splice_children(pos..pos + 1, vec![new_pattern_node.into()]);
        }
        // TODO: else insert new node after URL
    }

    /// Set the version policy of the entry.
    ///
    /// TODO: This currently only replaces an existing version policy.
    /// If the entry doesn't have a version policy, this method does nothing.
    /// Future implementation should insert the node at the correct position.
    pub fn set_version_policy(&mut self, new_policy: &str) {
        // Build the new VERSION_POLICY node
        let mut builder = GreenNodeBuilder::new();
        builder.start_node(VERSION_POLICY.into());
        // Version policy can be KEY (e.g., "debian") or VALUE
        builder.token(VALUE.into(), new_policy);
        builder.finish_node();
        let new_policy_green = builder.finish();

        // Create a syntax node (splice_children will detach and reattach it)
        let new_policy_node = SyntaxNode::new_root_mut(new_policy_green);

        // Find existing VERSION_POLICY node position
        let policy_pos = self.0.children_with_tokens().position(
            |child| matches!(child, SyntaxElement::Node(node) if node.kind() == VERSION_POLICY),
        );

        if let Some(pos) = policy_pos {
            // Replace existing VERSION_POLICY node
            self.0
                .splice_children(pos..pos + 1, vec![new_policy_node.into()]);
        }
        // TODO: else insert new node after MATCHING_PATTERN (or URL if no pattern)
    }

    /// Set the script of the entry.
    ///
    /// TODO: This currently only replaces an existing script.
    /// If the entry doesn't have a script, this method does nothing.
    /// Future implementation should insert the node at the correct position.
    pub fn set_script(&mut self, new_script: &str) {
        // Build the new SCRIPT node
        let mut builder = GreenNodeBuilder::new();
        builder.start_node(SCRIPT.into());
        // Script can be KEY (e.g., "uupdate") or VALUE
        builder.token(VALUE.into(), new_script);
        builder.finish_node();
        let new_script_green = builder.finish();

        // Create a syntax node (splice_children will detach and reattach it)
        let new_script_node = SyntaxNode::new_root_mut(new_script_green);

        // Find existing SCRIPT node position
        let script_pos = self
            .0
            .children_with_tokens()
            .position(|child| matches!(child, SyntaxElement::Node(node) if node.kind() == SCRIPT));

        if let Some(pos) = script_pos {
            // Replace existing SCRIPT node
            self.0
                .splice_children(pos..pos + 1, vec![new_script_node.into()]);
        }
        // TODO: else insert new node after VERSION_POLICY (or MATCHING_PATTERN/URL if no policy)
    }
}

const SUBSTITUTIONS: &[(&str, &str)] = &[
    // This is substituted with the source package name found in the first line
    // of the debian/changelog file.
    // "@PACKAGE@": None,
    // This is substituted by the legal upstream version regex (capturing).
    ("@ANY_VERSION@", r"[-_]?(\d[\-+\.:\~\da-zA-Z]*)"),
    // This is substituted by the typical archive file extension regex
    // (non-capturing).
    (
        "@ARCHIVE_EXT@",
        r"(?i)\.(?:tar\.xz|tar\.bz2|tar\.gz|zip|tgz|tbz|txz)",
    ),
    // This is substituted by the typical signature file extension regex
    // (non-capturing).
    (
        "@SIGNATURE_EXT@",
        r"(?i)\.(?:tar\.xz|tar\.bz2|tar\.gz|zip|tgz|tbz|txz)\.(?:asc|pgp|gpg|sig|sign)",
    ),
    // This is substituted by the typical Debian extension regexp (capturing).
    ("@DEB_EXT@", r"[\+~](debian|dfsg|ds|deb)(\.)?(\d+)?$"),
];

pub fn subst(text: &str, package: impl FnOnce() -> String) -> String {
    let mut substs = SUBSTITUTIONS.to_vec();
    let package_name;
    if text.contains("@PACKAGE@") {
        package_name = Some(package());
        substs.push(("@PACKAGE@", package_name.as_deref().unwrap()));
    }

    let mut text = text.to_string();

    for (k, v) in substs {
        text = text.replace(k, v);
    }

    text
}

#[test]
fn test_subst() {
    assert_eq!(
        subst("@ANY_VERSION@", || unreachable!()),
        r"[-_]?(\d[\-+\.:\~\da-zA-Z]*)"
    );
    assert_eq!(subst("@PACKAGE@", || "dulwich".to_string()), "dulwich");
}

impl OptionList {
    fn children(&self) -> impl Iterator<Item = _Option> + '_ {
        self.0.children().filter_map(_Option::cast)
    }

    pub fn has_option(&self, key: &str) -> bool {
        self.children().any(|it| it.key().as_deref() == Some(key))
    }

    pub fn get_option(&self, key: &str) -> Option<String> {
        for child in self.children() {
            if child.key().as_deref() == Some(key) {
                return child.value();
            }
        }
        None
    }
}

impl _Option {
    /// Returns the key of the option.
    pub fn key(&self) -> Option<String> {
        self.0.children_with_tokens().find_map(|it| match it {
            SyntaxElement::Token(token) => {
                if token.kind() == KEY {
                    Some(token.text().to_string())
                } else {
                    None
                }
            }
            _ => None,
        })
    }

    /// Returns the value of the option.
    pub fn value(&self) -> Option<String> {
        self.0
            .children_with_tokens()
            .filter_map(|it| match it {
                SyntaxElement::Token(token) => {
                    if token.kind() == VALUE || token.kind() == KEY {
                        Some(token.text().to_string())
                    } else {
                        None
                    }
                }
                _ => None,
            })
            .nth(1)
    }
}

impl Url {
    /// Returns the URL string.
    pub fn url(&self) -> String {
        self.0
            .children_with_tokens()
            .find_map(|it| match it {
                SyntaxElement::Token(token) => {
                    if token.kind() == VALUE {
                        Some(token.text().to_string())
                    } else {
                        None
                    }
                }
                _ => None,
            })
            .unwrap()
    }
}

impl MatchingPattern {
    /// Returns the matching pattern string.
    pub fn pattern(&self) -> String {
        self.0
            .children_with_tokens()
            .find_map(|it| match it {
                SyntaxElement::Token(token) => {
                    if token.kind() == VALUE {
                        Some(token.text().to_string())
                    } else {
                        None
                    }
                }
                _ => None,
            })
            .unwrap()
    }
}

impl VersionPolicyNode {
    /// Returns the version policy string.
    pub fn policy(&self) -> String {
        self.0
            .children_with_tokens()
            .find_map(|it| match it {
                SyntaxElement::Token(token) => {
                    // Can be KEY (e.g., "debian") or VALUE
                    if token.kind() == VALUE || token.kind() == KEY {
                        Some(token.text().to_string())
                    } else {
                        None
                    }
                }
                _ => None,
            })
            .unwrap()
    }
}

impl ScriptNode {
    /// Returns the script string.
    pub fn script(&self) -> String {
        self.0
            .children_with_tokens()
            .find_map(|it| match it {
                SyntaxElement::Token(token) => {
                    // Can be KEY (e.g., "uupdate") or VALUE
                    if token.kind() == VALUE || token.kind() == KEY {
                        Some(token.text().to_string())
                    } else {
                        None
                    }
                }
                _ => None,
            })
            .unwrap()
    }
}

#[test]
fn test_entry_node_structure() {
    // Test that entries properly use the new node types
    let wf: super::WatchFile = r#"version=4
opts=compression=xz https://example.com/releases (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let entry = wf.entries().next().unwrap();

    // Verify URL node exists and works
    assert_eq!(entry.0.children().find(|n| n.kind() == URL).is_some(), true);
    assert_eq!(entry.url(), "https://example.com/releases");

    // Verify MATCHING_PATTERN node exists and works
    assert_eq!(
        entry
            .0
            .children()
            .find(|n| n.kind() == MATCHING_PATTERN)
            .is_some(),
        true
    );
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );

    // Verify VERSION_POLICY node exists and works
    assert_eq!(
        entry
            .0
            .children()
            .find(|n| n.kind() == VERSION_POLICY)
            .is_some(),
        true
    );
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));

    // Verify SCRIPT node exists and works
    assert_eq!(
        entry.0.children().find(|n| n.kind() == SCRIPT).is_some(),
        true
    );
    assert_eq!(entry.script(), Some("uupdate".into()));
}

#[test]
fn test_entry_node_structure_partial() {
    // Test entry with only URL and pattern (no version or script)
    let wf: super::WatchFile = r#"version=4
https://github.com/example/tags .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let entry = wf.entries().next().unwrap();

    // Should have URL and MATCHING_PATTERN nodes
    assert_eq!(entry.0.children().find(|n| n.kind() == URL).is_some(), true);
    assert_eq!(
        entry
            .0
            .children()
            .find(|n| n.kind() == MATCHING_PATTERN)
            .is_some(),
        true
    );

    // Should NOT have VERSION_POLICY or SCRIPT nodes
    assert_eq!(
        entry
            .0
            .children()
            .find(|n| n.kind() == VERSION_POLICY)
            .is_some(),
        false
    );
    assert_eq!(
        entry.0.children().find(|n| n.kind() == SCRIPT).is_some(),
        false
    );

    // Verify accessors work correctly
    assert_eq!(entry.url(), "https://github.com/example/tags");
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(None));
    assert_eq!(entry.script(), None);
}

#[test]
fn test_parse_v1() {
    const WATCHV1: &str = r#"version=4
opts=bare,filenamemangle=s/.+\/v?(\d\S+)\.tar\.gz/syncthing-gtk-$1\.tar\.gz/ \
  https://github.com/syncthing/syncthing-gtk/tags .*/v?(\d\S+)\.tar\.gz
"#;
    let parsed = parse(WATCHV1);
    //assert_eq!(parsed.errors, Vec::<String>::new());
    let node = parsed.syntax();
    assert_eq!(
        format!("{:#?}", node),
        r#"ROOT@0..161
  VERSION@0..10
    KEY@0..7 "version"
    EQUALS@7..8 "="
    VALUE@8..9 "4"
    NEWLINE@9..10 "\n"
  ENTRY@10..161
    OPTS_LIST@10..86
      KEY@10..14 "opts"
      EQUALS@14..15 "="
      OPTION@15..19
        KEY@15..19 "bare"
      COMMA@19..20 ","
      OPTION@20..86
        KEY@20..34 "filenamemangle"
        EQUALS@34..35 "="
        VALUE@35..86 "s/.+\\/v?(\\d\\S+)\\.tar\\ ..."
    WHITESPACE@86..87 " "
    CONTINUATION@87..89 "\\\n"
    WHITESPACE@89..91 "  "
    URL@91..138
      VALUE@91..138 "https://github.com/sy ..."
    WHITESPACE@138..139 " "
    MATCHING_PATTERN@139..160
      VALUE@139..160 ".*/v?(\\d\\S+)\\.tar\\.gz"
    NEWLINE@160..161 "\n"
"#
    );

    let root = parsed.root();
    assert_eq!(root.version(), 4);
    let entries = root.entries().collect::<Vec<_>>();
    assert_eq!(entries.len(), 1);
    let entry = &entries[0];
    assert_eq!(
        entry.url(),
        "https://github.com/syncthing/syncthing-gtk/tags"
    );
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(None));
    assert_eq!(entry.script(), None);

    assert_eq!(node.text(), WATCHV1);
}

#[test]
fn test_parse_v2() {
    let parsed = parse(
        r#"version=4
https://github.com/syncthing/syncthing-gtk/tags .*/v?(\d\S+)\.tar\.gz
# comment
"#,
    );
    assert_eq!(parsed.errors, Vec::<String>::new());
    let node = parsed.syntax();
    assert_eq!(
        format!("{:#?}", node),
        r###"ROOT@0..90
  VERSION@0..10
    KEY@0..7 "version"
    EQUALS@7..8 "="
    VALUE@8..9 "4"
    NEWLINE@9..10 "\n"
  ENTRY@10..80
    URL@10..57
      VALUE@10..57 "https://github.com/sy ..."
    WHITESPACE@57..58 " "
    MATCHING_PATTERN@58..79
      VALUE@58..79 ".*/v?(\\d\\S+)\\.tar\\.gz"
    NEWLINE@79..80 "\n"
  COMMENT@80..89 "# comment"
  NEWLINE@89..90 "\n"
"###
    );

    let root = parsed.root();
    assert_eq!(root.version(), 4);
    let entries = root.entries().collect::<Vec<_>>();
    assert_eq!(entries.len(), 1);
    let entry = &entries[0];
    assert_eq!(
        entry.url(),
        "https://github.com/syncthing/syncthing-gtk/tags"
    );
    assert_eq!(
        entry.format_url(|| "syncthing-gtk".to_string()),
        "https://github.com/syncthing/syncthing-gtk/tags"
            .parse()
            .unwrap()
    );
}

#[test]
fn test_parse_v3() {
    let parsed = parse(
        r#"version=4
https://github.com/syncthing/@PACKAGE@/tags .*/v?(\d\S+)\.tar\.gz
# comment
"#,
    );
    assert_eq!(parsed.errors, Vec::<String>::new());
    let root = parsed.root();
    assert_eq!(root.version(), 4);
    let entries = root.entries().collect::<Vec<_>>();
    assert_eq!(entries.len(), 1);
    let entry = &entries[0];
    assert_eq!(entry.url(), "https://github.com/syncthing/@PACKAGE@/tags");
    assert_eq!(
        entry.format_url(|| "syncthing-gtk".to_string()),
        "https://github.com/syncthing/syncthing-gtk/tags"
            .parse()
            .unwrap()
    );
}

#[test]
fn test_parse_v4() {
    let cl: super::WatchFile = r#"version=4
opts=repack,compression=xz,dversionmangle=s/\+ds//,repacksuffix=+ds \
    https://github.com/example/example-cat/tags \
        (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();
    assert_eq!(cl.version(), 4);
    let entries = cl.entries().collect::<Vec<_>>();
    assert_eq!(entries.len(), 1);
    let entry = &entries[0];
    assert_eq!(entry.url(), "https://github.com/example/example-cat/tags");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert!(entry.repack());
    assert_eq!(entry.compression(), Ok(Some(Compression::Xz)));
    assert_eq!(entry.dversionmangle(), Some("s/\\+ds//".into()));
    assert_eq!(entry.repacksuffix(), Some("+ds".into()));
    assert_eq!(entry.script(), Some("uupdate".into()));
    assert_eq!(
        entry.format_url(|| "example-cat".to_string()),
        "https://github.com/example/example-cat/tags"
            .parse()
            .unwrap()
    );
    assert_eq!(entry.version(), Ok(Some(VersionPolicy::Debian)));
}

#[test]
fn test_git_mode() {
    let text = r#"version=3
opts="mode=git, gitmode=shallow, pgpmode=gittag" \
https://git.kernel.org/pub/scm/linux/kernel/git/firmware/linux-firmware.git \
refs/tags/(.*) debian
"#;
    let parsed = parse(text);
    assert_eq!(parsed.errors, Vec::<String>::new());
    let cl = parsed.root();
    assert_eq!(cl.version(), 3);
    let entries = cl.entries().collect::<Vec<_>>();
    assert_eq!(entries.len(), 1);
    let entry = &entries[0];
    assert_eq!(
        entry.url(),
        "https://git.kernel.org/pub/scm/linux/kernel/git/firmware/linux-firmware.git"
    );
    assert_eq!(entry.matching_pattern(), Some("refs/tags/(.*)".into()));
    assert_eq!(entry.version(), Ok(Some(VersionPolicy::Debian)));
    assert_eq!(entry.script(), None);
    assert_eq!(entry.gitmode(), Ok(GitMode::Shallow));
    assert_eq!(entry.pgpmode(), Ok(PgpMode::GitTag));
    assert_eq!(entry.mode(), Ok(Mode::Git));
}

#[test]
fn test_parse_quoted() {
    const WATCHV1: &str = r#"version=4
opts="bare, filenamemangle=blah" \
  https://github.com/syncthing/syncthing-gtk/tags .*/v?(\d\S+)\.tar\.gz
"#;
    let parsed = parse(WATCHV1);
    //assert_eq!(parsed.errors, Vec::<String>::new());
    let node = parsed.syntax();

    let root = parsed.root();
    assert_eq!(root.version(), 4);
    let entries = root.entries().collect::<Vec<_>>();
    assert_eq!(entries.len(), 1);
    let entry = &entries[0];

    assert_eq!(
        entry.url(),
        "https://github.com/syncthing/syncthing-gtk/tags"
    );
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(None));
    assert_eq!(entry.script(), None);

    assert_eq!(node.text(), WATCHV1);
}

#[test]
fn test_set_url() {
    // Test setting URL on a simple entry without options
    let wf: super::WatchFile = r#"version=4
https://github.com/syncthing/syncthing-gtk/tags .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(
        entry.url(),
        "https://github.com/syncthing/syncthing-gtk/tags"
    );

    entry.set_url("https://newurl.example.org/path");
    assert_eq!(entry.url(), "https://newurl.example.org/path");
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "https://newurl.example.org/path .*/v?(\\d\\S+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_set_url_with_options() {
    // Test setting URL on an entry with options
    let wf: super::WatchFile = r#"version=4
opts=foo=blah https://foo.com/bar .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.url(), "https://foo.com/bar");
    assert_eq!(entry.get_option("foo"), Some("blah".to_string()));

    entry.set_url("https://example.com/baz");
    assert_eq!(entry.url(), "https://example.com/baz");

    // Verify options are preserved
    assert_eq!(entry.get_option("foo"), Some("blah".to_string()));
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "opts=foo=blah https://example.com/baz .*/v?(\\d\\S+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_set_url_complex() {
    // Test with a complex watch file with multiple options and continuation
    let wf: super::WatchFile = r#"version=4
opts=bare,filenamemangle=s/.+\/v?(\d\S+)\.tar\.gz/syncthing-gtk-$1\.tar\.gz/ \
  https://github.com/syncthing/syncthing-gtk/tags .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(
        entry.url(),
        "https://github.com/syncthing/syncthing-gtk/tags"
    );

    entry.set_url("https://gitlab.com/newproject/tags");
    assert_eq!(entry.url(), "https://gitlab.com/newproject/tags");

    // Verify all options are preserved
    assert!(entry.bare());
    assert_eq!(
        entry.filenamemangle(),
        Some("s/.+\\/v?(\\d\\S+)\\.tar\\.gz/syncthing-gtk-$1\\.tar\\.gz/".into())
    );
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );

    // Verify the exact serialized output preserves structure
    assert_eq!(
        entry.to_string(),
        r#"opts=bare,filenamemangle=s/.+\/v?(\d\S+)\.tar\.gz/syncthing-gtk-$1\.tar\.gz/ \
  https://gitlab.com/newproject/tags .*/v?(\d\S+)\.tar\.gz
"#
    );
}

#[test]
fn test_set_url_with_all_fields() {
    // Test with all fields: options, URL, matching pattern, version, and script
    let wf: super::WatchFile = r#"version=4
opts=repack,compression=xz,dversionmangle=s/\+ds//,repacksuffix=+ds \
    https://github.com/example/example-cat/tags \
        (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.url(), "https://github.com/example/example-cat/tags");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));
    assert_eq!(entry.script(), Some("uupdate".into()));

    entry.set_url("https://gitlab.example.org/project/releases");
    assert_eq!(entry.url(), "https://gitlab.example.org/project/releases");

    // Verify all other fields are preserved
    assert!(entry.repack());
    assert_eq!(entry.compression(), Ok(Some(super::Compression::Xz)));
    assert_eq!(entry.dversionmangle(), Some("s/\\+ds//".into()));
    assert_eq!(entry.repacksuffix(), Some("+ds".into()));
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));
    assert_eq!(entry.script(), Some("uupdate".into()));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        r#"opts=repack,compression=xz,dversionmangle=s/\+ds//,repacksuffix=+ds \
    https://gitlab.example.org/project/releases \
        (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    );
}

#[test]
fn test_set_url_quoted_options() {
    // Test with quoted options
    let wf: super::WatchFile = r#"version=4
opts="bare, filenamemangle=blah" \
  https://github.com/syncthing/syncthing-gtk/tags .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(
        entry.url(),
        "https://github.com/syncthing/syncthing-gtk/tags"
    );

    entry.set_url("https://example.org/new/path");
    assert_eq!(entry.url(), "https://example.org/new/path");

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        r#"opts="bare, filenamemangle=blah" \
  https://example.org/new/path .*/v?(\d\S+)\.tar\.gz
"#
    );
}

#[test]
fn test_set_matching_pattern() {
    // Test setting matching pattern on a simple entry
    let wf: super::WatchFile = r#"version=4
https://github.com/example/tags .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );

    entry.set_matching_pattern("(?:.*?/)?v?([\\d.]+)\\.tar\\.gz");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?([\\d.]+)\\.tar\\.gz".into())
    );

    // Verify URL is preserved
    assert_eq!(entry.url(), "https://github.com/example/tags");

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "https://github.com/example/tags (?:.*?/)?v?([\\d.]+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_set_matching_pattern_with_all_fields() {
    // Test with all fields present
    let wf: super::WatchFile = r#"version=4
opts=compression=xz https://example.com/releases (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );

    entry.set_matching_pattern(".*/version-([\\d.]+)\\.tar\\.xz");
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/version-([\\d.]+)\\.tar\\.xz".into())
    );

    // Verify all other fields are preserved
    assert_eq!(entry.url(), "https://example.com/releases");
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));
    assert_eq!(entry.script(), Some("uupdate".into()));
    assert_eq!(entry.compression(), Ok(Some(super::Compression::Xz)));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "opts=compression=xz https://example.com/releases .*/version-([\\d.]+)\\.tar\\.xz debian uupdate\n"
    );
}

#[test]
fn test_set_version_policy() {
    // Test setting version policy
    let wf: super::WatchFile = r#"version=4
https://example.com/releases (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));

    entry.set_version_policy("previous");
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Previous)));

    // Verify all other fields are preserved
    assert_eq!(entry.url(), "https://example.com/releases");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert_eq!(entry.script(), Some("uupdate".into()));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "https://example.com/releases (?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz previous uupdate\n"
    );
}

#[test]
fn test_set_version_policy_with_options() {
    // Test with options and continuation
    let wf: super::WatchFile = r#"version=4
opts=repack,compression=xz \
    https://github.com/example/example-cat/tags \
        (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));

    entry.set_version_policy("ignore");
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Ignore)));

    // Verify all other fields are preserved
    assert_eq!(entry.url(), "https://github.com/example/example-cat/tags");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert_eq!(entry.script(), Some("uupdate".into()));
    assert!(entry.repack());

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        r#"opts=repack,compression=xz \
    https://github.com/example/example-cat/tags \
        (?:.*?/)?v?(\d[\d.]*)\.tar\.gz ignore uupdate
"#
    );
}

#[test]
fn test_set_script() {
    // Test setting script
    let wf: super::WatchFile = r#"version=4
https://example.com/releases (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.script(), Some("uupdate".into()));

    entry.set_script("uscan");
    assert_eq!(entry.script(), Some("uscan".into()));

    // Verify all other fields are preserved
    assert_eq!(entry.url(), "https://example.com/releases");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "https://example.com/releases (?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz debian uscan\n"
    );
}

#[test]
fn test_set_script_with_options() {
    // Test with options
    let wf: super::WatchFile = r#"version=4
opts=compression=xz https://example.com/releases (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.script(), Some("uupdate".into()));

    entry.set_script("custom-script.sh");
    assert_eq!(entry.script(), Some("custom-script.sh".into()));

    // Verify all other fields are preserved
    assert_eq!(entry.url(), "https://example.com/releases");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));
    assert_eq!(entry.compression(), Ok(Some(super::Compression::Xz)));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "opts=compression=xz https://example.com/releases (?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz debian custom-script.sh\n"
    );
}
