/*
 * Copyright 2026 Arnaud Rebillout
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <gio/gio.h>
#include <xcb/xcb.h>
#include <xcb/randr.h>

// Time to wait between the last event and the actual resize, in ms
#define LATENCY 500

GMainLoop *loop;
guint resize_timeout_id;
xcb_connection_t *conn;

xcb_screen_t *default_screen(void) {
	const xcb_setup_t *setup;
	xcb_screen_iterator_t iter;
	xcb_screen_t *screen;

	setup = xcb_get_setup(conn);
	iter = xcb_setup_roots_iterator(setup);
	screen = iter.data;

	return screen;
}

void resize(gpointer user_data) {
	int res;

	res = system("xrandr -s 0");
	if (res == 0) {
		g_debug("Resized!");
	} else {
		g_warning("Failed to resize");
	}

	resize_timeout_id = 0;
}

gboolean handle_x_event(GIOChannel *source, GIOCondition condition, gpointer data) {
	xcb_generic_event_t *evt;
	xcb_randr_screen_change_notify_event_t *randr_evt;
	gboolean need_resize = FALSE;

	g_debug("-> handle_x_event: condition=%d", condition);

	// Note: we receive (G_IO_IN & G_IO_HUP) when users log out
	if (condition & (G_IO_ERR | G_IO_HUP)) {
		g_debug("Some error happened, quitting");
		g_main_loop_quit(loop);
		return FALSE;
	}

	while ((evt = xcb_poll_for_event(conn)) != NULL) {
		g_debug("Event type: %d", evt->response_type);

		if (evt->response_type == 0)
			continue;

		need_resize = TRUE;

		randr_evt = (xcb_randr_screen_change_notify_event_t *) evt;
		g_debug("Timestamp: %u", randr_evt->timestamp);
		g_debug("Screen size: %u x %u", randr_evt->width, randr_evt->height);

		free(evt);
	}

	if (need_resize == TRUE) {
		g_clear_handle_id(&resize_timeout_id, g_source_remove);
		resize_timeout_id = g_timeout_add_once(LATENCY, resize, NULL);
	}

	return TRUE;
}

void setup(gpointer user_data) {
	GIOChannel *channel;
	int fd;

	// Subscribe to screen change events
	xcb_randr_select_input(conn, default_screen()->root,
			XCB_RANDR_NOTIFY_MASK_SCREEN_CHANGE);
	xcb_flush(conn);

	// Get the file descriptor of the connection
	fd = xcb_get_file_descriptor(conn);
	if (fd == -1) {
		g_warning("Failed to get xcb fd");
		g_main_loop_quit(loop);
	}

	// Add a GIO watch for this file descriptor
	channel = g_io_channel_unix_new(fd);
	g_io_channel_set_encoding(channel, NULL, NULL);
	g_io_add_watch(channel, G_IO_ERR | G_IO_HUP | G_IO_IN | G_IO_PRI, handle_x_event, NULL);
	g_io_channel_unref(channel);
}

int main(int argc, char *argv[]) {
	conn = xcb_connect(NULL, NULL);
	if (xcb_connection_has_error(conn)) {
		fprintf(stderr, "xcb_connect failed\n");
		xcb_disconnect(conn);
		return EXIT_FAILURE;
	}

	loop = g_main_loop_new(NULL, FALSE);
	g_idle_add_once(setup, loop);

	g_debug("Start main loop");
	g_main_loop_run(loop);
	g_debug("Main loop ended");

	g_main_loop_unref(loop);
	xcb_disconnect(conn);

	return EXIT_SUCCESS;
}
